"""
This file is part of Totara Enterprise Extensions.

Copyright (C) 2021 onward Totara Learning Solutions LTD

Totara Enterprise Extensions is provided only to Totara
Learning Solutions LTD's customers and partners, pursuant to
the terms and conditions of a separate agreement with Totara
Learning Solutions LTD or its affiliate.

If you do not have an agreement with Totara Learning Solutions
LTD, you may not access, use, modify, or distribute this software.
Please contact [licensing@totaralearning.com] for more information.

@author Cody Finegan <cody.finegan@totaralearning.com>
@package ml_service
"""

import hashlib
import time
from werkzeug.wrappers import Request, Response


class AuthenticationMiddleware:
    """
    Middleware to check the request is valid & acceptable
    """

    def __init__(self, app, config):
        """
        Class constructor method

        :param app: The Flask object
        :type app: Flask
        :param config: An instance of the class that contains base configurations of
            the ML Service
        :type config: An instance of configuration class
        """
        self.app = app
        self.config = config
        self.skip = [
            "/favicon.ico",
            "/",
        ]
        self.config["auth_info"] = {}

    def __call__(self, environ, start_response):
        """
        This allows the instances of the class to be run as functions. The requests
        that are made on the endpoints mentioned in skip parameter of the class or
        have valid authentication headers are forwarded to the specific endpoints.
        The requests that don't have valid authentication headers don't pass from this
        layer and are responded with status code 401 and a message

        :param environ: The WSGI environ is generated by the WSGI server and contains
            information about the server configuration and client request
        :type environ: WSGIEnvironment
        :param start_response: The response callable provided by the WSGI server
        :type start_response: StartResponse
        :return: An application iterator
        :rtype: Iterable[bytes]
        """
        request = Request(environ)

        relative_url = "/" + request.url.replace(request.url_root, "")
        if relative_url in self.skip:
            return self.app(environ, start_response)

        # Check both the user agent & key is expected.
        # Note, we currently only support the Totara CURL client, if that changes
        # then the user agent check here must also change
        request_user_agent = request.headers.get("User-Agent", "")
        request_totara_key = request.headers.get("X-Totara-Ml-Key", "")
        request_totara_time = request.headers.get("X-Totara-Time", "0")

        valid_user_agent = (
            request_user_agent == "TotaraBot/1.0"
            or self.config.get("APP_MODE", "") == "Development"
        )
        valid_timestamp = request_totara_time and self.validate_timestamp(
            request_totara_time
        )
        valid_key = request_totara_key and self.validate_key(
            request_totara_key, request_totara_time
        )

        if valid_user_agent and valid_key and valid_timestamp:
            return self.app(environ, start_response)

        message = "Invalid access\n"
        if not valid_timestamp:
            message += "The clocks were not in sync\n"

        message += "\n" + str(request.headers)

        for key, value in self.config["auth_info"].items():
            message += f"\n{key}: {value}"

        res_builder = Response(bytes(message.encode("utf-8")), status=401)
        return res_builder(environ, start_response)

    def validate_timestamp(self, request_timestamp) -> bool:
        """
        Validate the timestamp provided is recent to a given acceptable variance

        :param request_timestamp: The timestamp received in the headers of the request
        :type request_timestamp: str
        :return: Whether the timestamp is valid or not
        :rtype: bool
        """
        current_timestamp = int(time.time())
        self.info("service_time", current_timestamp)
        rq_time = int(request_timestamp)

        diff = max(current_timestamp, rq_time) - min(current_timestamp, rq_time)
        self.info("variance", diff)

        return 30 > diff > -30

    def validate_key(self, request_key, request_timestamp) -> bool:
        """
        Validate the request key matches what we expected.

        :param request_key: The hashed key received in headers of the request
        :type request_key: str
        :param request_timestamp: The timestamp received in headers of the request
        :type request_timestamp: str
        :return: Whether the key is valid or not
        :rtype: bool
        """
        totara_key = self.config.get("TOTARA_KEY") or ""
        expected = hashlib.sha256(
            str(request_timestamp + totara_key).encode("utf8")
        ).hexdigest()

        return totara_key and expected == request_key

    def info(self, key, value) -> None:
        """
        This adds key-value pair in config['auth_info'] parameter (dictionary) of the
        class

        :param key: Key at which the value needs to be added
        :type key: str
        :type value: Value that needs to be added on the specified key
        :type value: any obj
        """
        self.config["auth_info"][key] = value
