<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2010 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Eugene Venter <eugene@catalyst.net.nz>
 * @package totara
 * @subpackage totara_sync
 */

global $CFG;
require_once($CFG->dirroot.'/admin/tool/totara_sync/elements/classes/element.class.php');
require_once($CFG->dirroot.'/totara/customfield/fieldlib.php');

abstract class totara_sync_hierarchy extends totara_sync_element {

    private $hierarchy;

    /**
     * Get hierarchy (implement in child class)
     *
     * @return stdClass the hierarchy object
     */
    abstract function get_hierarchy();

    function __construct() {
        parent::__construct();
        $this->hierarchy = $this->get_hierarchy();
    }

    function has_config() {
        return true;
    }

    /**
     * @param MoodleQuickForm $mform
     */
    function config_form(&$mform) {

        $mform->addElement('selectyesno', 'sourceallrecords', get_string('sourceallrecords', 'tool_totara_sync'));
        $mform->setDefault('sourceallrecords', 0);

        // Set the percentage of records relative to the number in the system below which the import task will ask for confirmation.
        $mform->addElement('text', 'minimum_records_threshold', get_string('minimum_records_threshold', 'tool_totara_sync'));
        $mform->setType('minimum_records_threshold', PARAM_INT);
        $mform->setDefault('minimum_records_threshold', $this->config->minimum_records_threshold ?? 100);
        $mform->addElement('static', 'minimum_records_threshold_description', '', get_string('minimum_records_threshold_description', 'tool_totara_sync'));

        $mform->addElement('header', 'crud', get_string('allowedactions', 'tool_totara_sync'));
        $mform->addElement('checkbox', 'allow_create', get_string('create', 'tool_totara_sync'));
        $mform->setDefault('allow_create', 1);
        $mform->addElement('checkbox', 'allow_update', get_string('update', 'tool_totara_sync'));
        $mform->setDefault('allow_update', 1);
        $mform->addElement('checkbox', 'allow_delete', get_string('delete', 'tool_totara_sync'));
        $mform->setDefault('allow_delete', 1);
        $mform->setExpanded('crud');
    }

    function config_save($data) {
        $this->set_config('sourceallrecords', $data->sourceallrecords);
        $this->set_config('minimum_records_threshold', $data->minimum_records_threshold);

        // Only set this config is we get some data. Vaiid values are 0 and 1, but if set
        // to null it can be used to determine whether CSV or database sync is being used.
        if (isset($data->csvsaveemptyfields)) {
            $this->set_config('csvsaveemptyfields', !empty($data->csvsaveemptyfields));
        } else {
            $this->set_config('csvsaveemptyfields', null);
        }

        $this->set_config('allow_create', !empty($data->allow_create));
        $this->set_config('allow_update', !empty($data->allow_update));
        $this->set_config('allow_delete', !empty($data->allow_delete));

        $sourcename = 'source_' . $this->get_name();

        if (!empty($data->$sourcename)) {
            $source = $this->get_source($data->$sourcename);
            // Build link to source config.
            $url = new moodle_url('/admin/tool/totara_sync/admin/sourcesettings.php', array('element' => $this->get_name(), 'source' => $source->get_name()));
            if ($source->has_config()) {
                // Set import_deleted and warn if necessary.
                $import_deleted_new = ($data->sourceallrecords == 0) ? '1' : '0';
                $import_deleted_old = $source->get_config('import_deleted');
                if ($import_deleted_new != $import_deleted_old) {
                    $source->set_config('import_deleted', $import_deleted_new);
                    \core\notification::warning(get_string('checkconfig', 'tool_totara_sync', $url->out()));
                }
            }
        }
    }

    function sync() {
        global $DB;

        $elname = $this->get_name();

        $this->addlog(get_string('syncstarted', 'tool_totara_sync'), 'info', "{$elname}sync");
        if (!$synctable = $this->get_source_sync_table()) {
            throw new totara_sync_exception($elname, "{$elname}sync", 'couldnotgetsourcetable');
        }

        // Create a clone of the temporary table
        if (!$synctable_clone = $this->get_source_sync_table_clone($synctable)) {
            throw new totara_sync_exception($elname, "{$elname}sync", 'couldnotcreateclonetable');
        }

        if (!$this->check_sanity($synctable, $synctable_clone)) {
            $this->get_source()->drop_table($synctable_clone);
            throw new totara_sync_exception($elname, "{$elname}sync", 'sanitycheckfailed');
        }

        // Initialise to safe defaults if settings not present.
        if (!isset($this->config->sourceallrecords)) {
            $this->config->sourceallrecords = 0;
        }
        if (!isset($this->config->allow_create)) {
            $this->config->allow_create = 0;
        }
        if (!isset($this->config->allow_update)) {
            $this->config->allow_update = 0;
        }
        if (!isset($this->config->allow_delete)) {
            $this->config->allow_delete = 0;
        }

        // Make sure the required deleted column is present if necessary.
        $synctablecolumns = $DB->get_columns($synctable);
        $deletedcolumnpresent = isset($synctablecolumns['deleted']);

        if ($this->config->sourceallrecords == 0 && !$deletedcolumnpresent) {
            throw new totara_sync_exception($elname, "{$elname}sync", 'deletefieldmissingnotallrecords');
        }

        // Create/update items
        if ($this->config->sourceallrecords == 1) {
            // If source contains all records then create/update
            // all records excluding obsolete/unmodified ones.
            $sql = "SELECT s.*
                      FROM {{$synctable}} s
                     WHERE s.idnumber NOT IN
                            (
                                SELECT ii.idnumber
                                  FROM {{$elname}} ii
                       LEFT OUTER JOIN {{$synctable_clone}} ss ON ii.idnumber = ss.idnumber
                                 WHERE (ii.totarasync = 1 AND ss.idnumber IS NULL)
                                    OR ss.timemodified = ii.timemodified
                            )";
        } else if ($this->config->sourceallrecords == 0 && $deletedcolumnpresent) {
            // If the source doesn't contain all records then we don't
            // want to create/update ones marked as deleted.
            // Note that the deleted column should always be present.
            $sql = "SELECT s.*
                      FROM {{$synctable}} s
                     WHERE s.idnumber NOT IN
                           (
                                SELECT ii.idnumber
                                  FROM {{$elname}} ii
                       LEFT OUTER JOIN {{$synctable_clone}} ss ON (ii.idnumber = ss.idnumber)
                                 WHERE (ii.totarasync = 1 AND ss.idnumber IS NULL AND ss.deleted = 1)
                                    OR ss.timemodified = ii.timemodified
                          )";
        }

        if (isset($sql)) {
            $rs = $DB->get_recordset_sql($sql);

            foreach ($rs as $item) {
                $this->sync_item($item, $synctable);
            }
        }

        /// Delete items.
        if (!empty($this->config->allow_delete)) {
            if ($this->config->sourceallrecords == 1) {
                // If the source contains all records then we delete
                // any items that don't exist in the source.
                $sql = "SELECT i.id, i.idnumber
                          FROM {{$elname}} i
               LEFT OUTER JOIN {{$synctable}} s ON i.idnumber = s.idnumber
                         WHERE i.totarasync = 1
                           AND s.idnumber IS NULL";
            } else {
                // If the deleted column is present then we delete items
                // marked as deleted.
                if ($deletedcolumnpresent) {
                    $sql = "SELECT i.id, i.idnumber
                              FROM {{$elname}} i
                   LEFT OUTER JOIN {{$synctable}} s ON i.idnumber = s.idnumber
                             WHERE i.totarasync = 1
                               AND s.deleted = 1";
                }
            }

            $rs = $DB->get_recordset_sql($sql);

            foreach ($rs as $r) {
                if (!$this->hierarchy->delete_hierarchy_item($r->id)) {
                    $this->addlog(get_string('cannotdeletex', 'tool_totara_sync',
                        "{$elname} {$r->idnumber}"), 'warn', "{$elname}sync");
                } else {
                    $this->addlog(get_string('deletedx', 'tool_totara_sync',
                        "{$elname} {$r->idnumber}"), 'info', "{$elname}sync");
                }
            }
        }

        $this->get_source()->drop_table();
        $this->addlog(get_string('syncfinished', 'tool_totara_sync'), 'info', "{$elname}sync");

        return true;
    }

    /**
     * Sync an item
     *
     * @param stdClass $newitem object with escaped values
     * @param string $synctable sync table name
     * @return bool true because someone didn't like calling return without a value
     * @throws totara_sync_exception
     */
    function sync_item($newitem, $synctable) {
        global $DB;

        if (empty($this->config->allow_create) && empty($this->config->allow_update)) {
            // not allowed to create/update, so return early
            return true;
        }

        $elname = $this->get_name();

        if ($newitem->frameworkidnumber === null) {
            // Ignore empty fields, set the old frameworkid.
            if (!$newitem->frameworkid = $DB->get_field($elname, 'frameworkid', array('idnumber' => $newitem->idnumber))) {
                throw new totara_sync_exception($elname, 'syncitem', 'existingitemxframeworkidnotfound',
                    $newitem->idnumber);
            }
        } else if (!$newitem->frameworkid = $DB->get_field("{$elname}_framework", 'id', array('idnumber' => $newitem->frameworkidnumber))) {
            throw new totara_sync_exception($elname, 'syncitem', 'frameworkxnotfound',
                $newitem->frameworkidnumber);
        }


        // null = keep current value
        // <number> = assign value
        //
        // Ensure newitem's parent is synced first - only non-existent or not already synced parent items.
        if ($newitem->parentidnumber !== null && $newitem->parentidnumber !== '') {
            // Cant find parent so try sync parent first
            if (!$parentid = $DB->get_field_select($elname, 'id', "idnumber = ?", array($newitem->parentidnumber))) {
                // Sync parent first (recursive)
                $sql = "SELECT *
                    FROM {{$synctable}}
                    WHERE idnumber = ? ";
                if (!$newparent = $DB->get_record_sql($sql, array($newitem->parentidnumber), IGNORE_MULTIPLE)) {
                    throw new totara_sync_exception($elname, 'syncitem', 'parentxnotfound',
                        $newitem->parentidnumber);
                }
                try {
                    $this->sync_item($newparent, $synctable);
                } catch (totara_sync_exception $e) {
                    throw new totara_sync_exception($elname, 'syncitem', 'cannotsyncitemparent',
                        $newitem->parentidnumber, $e->getMessage());
                }
                // Update parentid with the newly-created one
                $newitem->parentid = $DB->get_field($elname, 'id', array('idnumber' => $newitem->parentidnumber));
            } else {
                $newitem->parentid = $parentid;
            }
        } else if ($newitem->parentidnumber === null) {
            // If null then we keep the current value (if it exists).
            $newitem->parentid = $DB->get_field($elname, 'parentid', array('idnumber' => $newitem->idnumber));
        } else {
            // Zero will erase the existing parentid.
            $newitem->parentid = 0;
        }

        if (!isset($newitem->typeidnumber) && $newitem->typeidnumber === null) {
            // Use existing value.
            $typeid = $DB->get_field("{$elname}", 'typeid', array('idnumber' => $newitem->idnumber));
            $newitem->typeid = $typeid;
        } else if ($newitem->typeidnumber === 0) {
            // Delete
            $newitem->typeid = 0;
        } else {
            // Process new idnumber
            $newitem->typeid = $DB->get_field($elname.'_type', 'id', array('idnumber' => $newitem->typeidnumber));
        }

        // Unset the *idnumbers, since we now have the ids ;)
        unset($newitem->frameworkidnumber, $newitem->parentidnumber, $newitem->typeidnumber);

        if (!$dbitem = $DB->get_record($elname, array('idnumber' => $newitem->idnumber))) {  // TODO: make this scale
            ///
            /// Create new hierarchy item
            ///
            if (empty($this->config->allow_create)) {
                return true;
            }
            $newitem->totarasync = 1;
            $newitem->visible = 1;
            $newitem->usermodified = get_admin()->id;

            if (!$hitem = $this->hierarchy->add_hierarchy_item($newitem, $newitem->parentid, $newitem->frameworkid, true, true, false)) {
                throw new totara_sync_exception($elname, 'syncitem', 'cannotcreatex',
                    "{$elname} {$newitem->idnumber}");
            }

            // Save custom field data
            $newitem->id = $hitem->id;
            $this->sync_additional_data($newitem, false);

            $this->addlog(get_string('createdx', 'tool_totara_sync', "{$elname} {$hitem->idnumber}"), 'info', 'syncitem');

            return true;
        }

        if ((!empty($newitem->timemodified) && $newitem->timemodified == $dbitem->timemodified) || !$dbitem->totarasync) {
            // This record is not enabled for totara syncing OR
            // Modification time the same, we can skip this sync as nothing changed ;)
            return true;
        }

        $newitem->id = $dbitem->id;


        ///
        /// Update the item
        ///
        if (empty($this->config->allow_update)) {
            return true;
        }

        foreach ($newitem as $field => $value) {
            if ($value === null) {
                unset($newitem->$field); // Null means "don't update the existing data", so skip this field.
            }
        }

        $newitem->usermodified = get_admin()->id;
        if (!$this->hierarchy->update_hierarchy_item($dbitem->id, $newitem, false, true, false)) {
            throw new totara_sync_exception($elname, 'syncitem', 'cannotupdatex',
                "{$elname} {$newitem->idnumber}");
        }

        // Sync custom fields
        if (isset($newitem->typeid) && $newitem->typeid != $dbitem->typeid) {
            // Remove old custom field data
            $this->hierarchy->delete_custom_field_data($dbitem->id);
        }

        // Add/update custom field data
        $this->sync_additional_data($newitem, true);

        $this->addlog(get_string('updatedx', 'tool_totara_sync', "{$elname} {$newitem->idnumber}"), 'info', 'syncitem');

        return true;
    }

    /**
     * Save additional data relevant to the hierarchy item, such as custom fields
     *
     * @param object $data Hierarchy item data
     * @param bool $is_updating Are we updating or creating this hierarchy, or just creating?
     */
    protected function sync_additional_data(object $data, bool $is_updating) {
        if (empty($data->customfields) || ($is_updating && empty($data->typeid))) {
            return;
        }

        if ($custom_fields = json_decode($data->customfields)) {
            foreach ($custom_fields as $name => $value) {
                if ($value === null) {
                    continue; // Null means "don't update the existing data", so skip this field.
                }

                // TODO: TL-16855 Check saving of menu customfield so it default data when deleting value

                $data->{$name} = $value;
            }
            customfield_save_data($data, $this->hierarchy->prefix, $this->hierarchy->shortprefix . '_type', true);
        }
    }

    /**
     * Checks the temporary table for data integrity
     *
     * @global object $DB
     * @param string $synctable
     * @param string $synctable_clone name of the clone table
     * @return boolean
     */
    function check_sanity($synctable, $synctable_clone) {
        global $DB;

        if (!empty($this->config->sourceallrecords) && !$this->check_table_has_content($synctable)) {
            throw new totara_sync_exception(
                $this->get_name(),
                'importdata',
                'totara_sync:sync_aborted_due_to_empty_source'
            );
        }

        $elname = $this->get_name();

        /// Check frameworks
        $sql = "SELECT DISTINCT frameworkidnumber
                  FROM {{$synctable}}
                 WHERE frameworkidnumber NOT IN
                    (SELECT idnumber
                       FROM {{$elname}_framework}
                    )";
        $rs = $DB->get_recordset_sql($sql);
        if ($rs->valid()) {
            foreach ($rs as $r) {
                $this->addlog(get_string('frameworkxnotexist', 'tool_totara_sync', $r->frameworkidnumber), 'error', 'checksanity');
            }
            return false;
        }

        /// Check duplicate idnumbers and warn
        $sql = "SELECT idnumber
            FROM {{$synctable}}
            GROUP BY idnumber
                HAVING COUNT(*) > 1";
        $rs = $DB->get_recordset_sql($sql);
        if ($rs->valid()) {
            foreach ($rs as $r) {
                $this->addlog(get_string('duplicateidnumberx', 'tool_totara_sync', $r->idnumber), 'warn', 'checksanity');
            }
        }

        // Check parents.
        if (empty($this->config->sourceallrecords)) {
            $parentssql = " AND (s.parentidnumber NOT IN (SELECT idnumber FROM {{$synctable_clone}}))
                AND s.parentidnumber NOT IN (SELECT idnumber FROM {{$elname}})";
        } else {
            $parentssql = "AND s.parentidnumber NOT IN (SELECT idnumber FROM {{$synctable_clone}})";
        }

        $sql = "SELECT DISTINCT s.parentidnumber
                  FROM {{$synctable}} s
       LEFT OUTER JOIN {{$elname}} i
                    ON s.parentidnumber = i.idnumber
                 WHERE s.parentidnumber IS NOT NULL AND s.parentidnumber != '' " .
            $parentssql;

        $rs = $DB->get_recordset_sql($sql);
        if ($rs->valid()) {
            foreach ($rs as $r) {
                $lngstr = empty($this->config->sourceallrecords) ? 'parentxnotexist' : 'parentxnotexistinfile';
                $this->addlog(get_string($lngstr, 'tool_totara_sync', $r->parentidnumber), 'error', 'checksanity');
            }
            return false;
        }

        /// Check types
        $sql = "SELECT DISTINCT typeidnumber
                 FROM {{$synctable}}
                WHERE typeidnumber IS NOT NULL AND typeidnumber != '' AND typeidnumber != '0'
                  AND typeidnumber NOT IN
                     (SELECT idnumber FROM {{$elname}_type})";
        $rs = $DB->get_recordset_sql($sql);
        if ($rs->valid()) {
            foreach ($rs as $r) {
                $this->addlog(get_string('typexnotexist', 'tool_totara_sync', $r->typeidnumber), 'error', 'checksanity');
            }
            return false;
        }

        /// Check circular parent references
        /// A circular reference will never have a root node (parentid == NULL)
        /// We can determine CRs by eliminating the nodes of the valid trees
        $sql = "SELECT idnumber, parentidnumber
                  FROM {{$synctable}}";
        $nodes = $DB->get_records_sql_menu($sql);

        // If source does not contain all data we need to also include current parents.
        if (empty($this->config->sourceallrecords)) {
            $items = $DB->get_records($elname, array(), '', 'id, idnumber, parentid');
            foreach ($items as $item) {
                if (!isset($nodes[$item->idnumber])) {
                    $nodes[$item->idnumber] = ($item->parentid == 0) ? '' : $items[$item->parentid]->idnumber;
                }
            }
            unset($items);
        }

        // Start eliminating nodes from the valid trees
        // Start at the top so get all the root nodes (no parentid)
        $top_nodes_1 = array_keys($nodes, '', true);
        $top_nodes_2 = array_keys($nodes, '0', true);
        $top_nodes_3 = array_keys($nodes, 0, true);
        $top_nodes_4 = array_keys($nodes, null, true);

        // Merge top level nodes into one array
        $goodnodes = array_merge($top_nodes_1, $top_nodes_2, $top_nodes_3, $top_nodes_4);

        unset($top_nodes_1, $top_nodes_2, $top_nodes_3, $top_nodes_4);

        while (!empty($goodnodes)) {
            $newgoodnodes = array();
            foreach ($goodnodes as $nid) {
                // Unset good parentnodes
                unset($nodes[$nid]);

                // Get all good childnodes
                $newgoodnodes = array_merge($newgoodnodes, array_keys($nodes, $nid));
            }

            $goodnodes = $newgoodnodes;
        }

        // Remaining nodes mean we have circular refs!
        if (!empty($nodes)) {
            $this->addlog(get_string('circularreferror', 'tool_totara_sync',
                (object)array('naughtynodes' => implode(', ', array_keys($nodes)))), 'error', 'checksanity');
            return false;
        }

        /// Get all hierarchy records to be created/updated - exclude obsolete/unmodified items
        $sql = "SELECT s.*
                  FROM {{$synctable}} s
                 WHERE s.idnumber NOT IN
                    (SELECT ii.idnumber
                       FROM {{$elname}} ii
            LEFT OUTER JOIN {{$synctable_clone}} ss
                         ON (ii.idnumber = ss.idnumber)
                      WHERE ss.idnumber IS NULL
                         OR ss.timemodified = ii.timemodified
                    )";
        $rs = $DB->get_recordset_sql($sql);
        if ($rs->valid()) {
            foreach ($rs as $r) {
                /// Check custom fields
                if ($r->customfields !== null && $customfielddata = json_decode($r->customfields, true)) {
                    $customfielddata = array_map(
                        function($data) {
                            return $data === null ? '' : trim($data);
                        },
                        $customfielddata
                    );
                    $customfields = array_keys($customfielddata);
                    if (empty($r->typeidnumber)) {
                        $customfielddata = array_filter($customfielddata);
                        if (empty($customfielddata)) {
                            // Type and customfield data empty, so skip ;)
                            continue;
                        }
                        $this->addlog(get_string('customfieldsnotype', 'tool_totara_sync', "({$elname}:{$r->idnumber})"), 'error', 'checksanity');
                        $rs->close();
                        return false;
                    }
                    if (!$typeid = $DB->get_field($elname.'_type', 'id', array('idnumber' => $r->typeidnumber))) {
                        $this->addlog(get_string('typexnotfound', 'tool_totara_sync', $r->typeidnumber), 'error', 'checksanity');
                        $rs->close();
                        return false;
                    }
                    foreach ($customfields as $c) {
                        if (empty($customfielddata[$c])) {
                            // Don't check empty fields, as this might be another type's custom field
                            continue;
                        }
                        $shortname = str_replace('customfield_', '', $c);
                        if (!$DB->record_exists($elname.'_type_info_field', array('typeid' => $typeid, 'shortname' => $shortname))) {
                            $this->addlog(get_string('customfieldnotexist', 'tool_totara_sync',
                                (object)array('shortname' => $shortname, 'typeidnumber' => $r->typeidnumber)), 'error', 'checksanity');
                            $rs->close();
                            return false;
                        }
                    }
                }
            }
        }

        return true;
    }
}
