<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Matthias Bonk <matthias.bonk@totara.com>
 * @package core_badges
 */

namespace core_badges\helper;

use coding_exception;
use core\cipher\manager as cipher_manager;

/**
 * Helper methods for encrypting and decrypting strings.
 *
 * Used for handling backpack passwords.
 */
class encryption_helper {

    public const ITEM_TYPE_EXTERNAL = 'badge_external_backpack';

    public const ITEM_TYPE_USER = 'badge_backpack';

    /**
     * Encrypt a string
     *
     * @param string $clear_text
     * @param string $item_id
     * @param string $item_type
     * @return string
     */
    public static function encrypt_string(string $clear_text, string $item_id, string $item_type): string {
        $cipher = (new cipher_manager())->encrypt($clear_text, $item_id, $item_type);
        if ($cipher === false) {
            // If we fail to encrypt, return the original value
            return $clear_text;
        }
        return $cipher;
    }

    /**
     * Decrypt a string
     *
     * If the given string is not an encrypted one, it will just return the string as is.
     * If the given string is an encrypted one, but we fail to decrypt (because $item_id or
     * $item_type don't match), then it will return an empty string.
     *
     * @param string $cipher_text
     * @param string $item_id
     * @param string $item_type
     * @return string
     */
    public static function decrypt_string(string $cipher_text, string $item_id, string $item_type): string {
        $manager = new cipher_manager();

        // Only try to decrypt if it looks like an encrypted string.
        if ($manager->is_encrypted_value($cipher_text)) {
            try {
                /**
                 * There is a problem with the decryption where an encrypted string is wrongly detected,
                 * e.g. '123::hello' or '1::any_string'.
                 * It is a real possibility, so we have to work around the fact that decrypt() throws an exception in this case.
                 * We return the clear text string then.
                 * (This problem should be fixed at some point in TL-46048)
                 */
                $decrypted_text = $manager->decrypt($cipher_text, $item_id, $item_type);
            }
            catch (coding_exception $e) {
                return $cipher_text;
            }

            if ($decrypted_text !== false) {
                // Return decrypted text only if decryption succeeded.
                return $decrypted_text;
            }
            // If we failed to decrypt, it either wasn't encrypted, or $item_id/$item_type were bad.
            // So we can only return the plain text.
        }
        return $cipher_text;
    }

    /**
     * Use this for encrypting when it's unknown if the string already has been encrypted, avoiding double-encryption.
     *
     * @param string $text
     * @param string $item_id
     * @param string $item_type
     * @return string
     */
    public static function encrypt_string_if_not_already_encrypted(string $text, string $item_id, string $item_type): string {
        $decrypted = static::decrypt_string($text, $item_id, $item_type);
        if ($decrypted !== $text) {
            // It was already encrypted.
            return $text;
        }

        // It was not encrypted, so encrypt it.
        return static::encrypt_string($text, $item_id, $item_type);
    }
}