<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Matthias Bonk <matthias.bonk@totara.com>
 * @package core_badges
 */

use core_badges\helper\encryption_helper;
use core_phpunit\testcase;

class core_badges_encryption_helper_test extends testcase {

    public function test_encrypt_result_depends_on_all_params(): void {
        $encrypted1 = encryption_helper::encrypt_string('clear_text', '111', 'some_item_type');
        static::assertNotEmpty($encrypted1);
        static::assertNotEquals('clear_text', $encrypted1);

        $encrypted2 = encryption_helper::encrypt_string('other_text', '111', 'some_item_type');
        static::assertNotEmpty($encrypted2);
        static::assertNotEquals('other_text', $encrypted2);
        static::assertNotEquals($encrypted1, $encrypted2);

        $encrypted3 = encryption_helper::encrypt_string('clear_text', '222', 'some_item_type');
        static::assertNotEmpty($encrypted3);
        static::assertNotEquals('clear_text', $encrypted3);
        static::assertNotEquals($encrypted1, $encrypted3);

        $encrypted4 = encryption_helper::encrypt_string('clear_text', '111', 'other_item_type');
        static::assertNotEmpty($encrypted4);
        static::assertNotEquals('clear_text', $encrypted4);
        static::assertNotEquals($encrypted1, $encrypted4);
    }

    public function test_decrypt_successful(): void {
        $clear_text_password = 'n%Ql!]Q&&K7V,PEiE5t%)';
        $encrypted = encryption_helper::encrypt_string($clear_text_password, '111', 'some_item_type');
        static::assertNotEquals($clear_text_password, $encrypted);

        $decrypted = encryption_helper::decrypt_string($encrypted, '111', 'some_item_type');
        static::assertSame($clear_text_password, $decrypted);
    }

    public function test_decrypt_fails_for_wrong_item_id(): void {
        $clear_text_password = 'n%Ql!]Q&&K7V,PEiE5t%)';
        $encrypted = encryption_helper::encrypt_string($clear_text_password, '111', 'some_item_type');
        static::assertNotEquals($clear_text_password, $encrypted);

        $result = encryption_helper::decrypt_string($encrypted, '222', 'some_item_type');

        // Decryption failed because $item_id is not right. The encrypted string is returned.
        static::assertSame($encrypted, $result);
    }

    public function test_decrypt_fails_for_wrong_item_type(): void {
        $clear_text_password = 'n%Ql!]Q&&K7V,PEiE5t%)';
        $encrypted = encryption_helper::encrypt_string($clear_text_password, '111', 'some_item_type');
        static::assertNotEquals($clear_text_password, $encrypted);

        $result = encryption_helper::decrypt_string($encrypted, '111', 'other_item_type');

        // Decryption failed because $item_type is not right. The encrypted string is returned.
        static::assertSame($encrypted, $result);
    }

    public function test_decrypt_returns_clear_text_when_no_encryption_detected(): void {
        $clear_text_password = 'n%Ql!]Q&&K7V,PEiE5t%)';

        $result = encryption_helper::decrypt_string($clear_text_password, '999', 'some_item_type');
        static::assertSame($clear_text_password, $result);

        // item_id and item_type don't matter if we cannot decrypt.
        $result = encryption_helper::decrypt_string($clear_text_password, '666', 'other_item_type');
        static::assertSame($clear_text_password, $result);
    }

    // Make sure we don't run into the problem that is still to be addressed in TL-46048 (at the time of writing).
    public function test_decrypt_returns_clear_text_even_when_text_looks_encrypted(): void {
        $only_looks_encrypted = '123::hello';

        $result = encryption_helper::decrypt_string($only_looks_encrypted, '999', 'some_item_type');
        static::assertSame($only_looks_encrypted, $result);

        // item_id and item_type don't matter if we cannot decrypt.
        $result = encryption_helper::decrypt_string($only_looks_encrypted, '666', 'other_item_type');
        static::assertSame($only_looks_encrypted, $result);
    }

    public function test_encrypt_string_if_not_already_encrypted_for_clear_text(): void {
        $clear_text_password = 'n%Ql!]Q&&K7V,PEiE5t%)';
        $result = encryption_helper::encrypt_string_if_not_already_encrypted($clear_text_password, '999', 'some_item_type');
        static::assertNotEmpty($result);
        static::assertNotEquals($clear_text_password, $result);

        // Make sure it's actually encrypted, meaning we can decrypt it.
        $decrypted = encryption_helper::decrypt_string($result, '999', 'some_item_type');
        static::assertSame($clear_text_password, $decrypted);
    }

    public function test_encrypt_string_if_not_already_encrypted_for_already_encrypted_string(): void {
        $clear_text_password = 'n%Ql!]Q&&K7V,PEiE5t%)';
        $encrypted = encryption_helper::encrypt_string($clear_text_password, '999', 'some_item_type');

        $result = encryption_helper::encrypt_string_if_not_already_encrypted($encrypted, '999', 'some_item_type');

        // Should be unchanged (no double encryption).
        static::assertSame($encrypted, $result);

        // Can also be decrypted.
        $decrypted = encryption_helper::decrypt_string($result, '999', 'some_item_type');
        static::assertSame($clear_text_password, $decrypted);
    }
}