<?php
/*
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Angela Kuznetsova <angela.kuznetsova@totara.com>
 * @package block_comments
 */

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once $CFG->dirroot . '/comment/lib.php';

/**
 * Test for posting comments
 *
 */
class block_comments_block_comments_test extends \core_phpunit\testcase {
    /** @var stdClass */
    private $alice;

    /** @var stdClass */
    private $bob;

    /** @var stdClass */
    private $course;

    /** @var stdClass */
    private $wiki;

    protected function setUp(): void {
        $this->setAdminUser();

        // Create users.
        $this->alice = $this->getDataGenerator()->create_user();
        $this->bob = $this->getDataGenerator()->create_user();

        // Create course and wiki module.
        $this->course = $this->getDataGenerator()->create_course();
        $this->wiki = $this->getDataGenerator()->create_module('wiki', ['course' => $this->course->id]);
    }

    protected function tearDown(): void {
        $this->alice = null;
        $this->bob = null;
        $this->course = null;
        $this->wiki = null;
        parent::tearDown();
    }

    /**
     * @param string $pagetypepattern
     * @param int $parentcontextid
     * @return void
     * @throws dml_exception
     */
    private function add_comments_block(string $pagetypepattern, int $parentcontextid): void {
        global $DB;
        $record = (object) [
            'blockname' => 'comments',
            'parentcontextid' => $parentcontextid,
            'showinsubcontexts' => 0,
            'pagetypepattern' => $pagetypepattern,
            'defaultregion' => 'main',
            'defaultweight' => 0,
            'configdata' => null,
            'timecreated' => time(),
            'timemodified' => time(),
        ];
        $DB->insert_record('block_instances', $record);
    }

    public function test_own_profile_is_allowed(): void {
        $this->setUser($this->alice);
        $context = \context_user::instance($this->alice->id);
        $args = (object) ['context' => $context];
        $this->add_comments_block("user_profile", $context->id);
        $perm = block_comments_comment_permissions($args);
        $this->assertTrue($perm['post']);
        $this->assertTrue($perm['view']);
    }

    public function test_other_user_block_is_denied(): void {
        $this->setUser($this->bob);
        $context_alice = \context_user::instance($this->alice->id);
        $args = (object) ['context' => $context_alice];
        $this->add_comments_block("user_profile", $context_alice->id);
        $perm = block_comments_comment_permissions($args);
        $this->assertFalse($perm['post']);
        $this->assertFalse($perm['view']);
    }

    public function test_own_dashboard_is_allowed(): void {
        $this->setUser($this->alice);
        $context = \context_user::instance($this->alice->id);
        $args = (object) ['context' => $context];
        $this->add_comments_block("totara-dashboard-1", $context->id);
        $perm = block_comments_comment_permissions($args);
        $this->assertTrue($perm['post']);
        $this->assertTrue($perm['view']);
    }

    public function test_other_user_dashboard_block_is_denied(): void {
        $this->setUser($this->bob);
        $context_alice = \context_user::instance($this->alice->id);
        $args = (object) ['context' => $context_alice];
        $this->add_comments_block("totara-dashboard-1", $context_alice->id);
        $perm = block_comments_comment_permissions($args);
        $this->assertFalse($perm['post']);
        $this->assertFalse($perm['view']);
    }

    public function test_comment_on_course(): void {
        $context = context_course::instance($this->course->id);
        $this->add_comments_block("course-view-*", $context->id);
        $this->setUser($this->alice);
        $context_alice = \context_user::instance($this->alice->id);
        $args = (object) ['context' => $context_alice];
        $perm = block_comments_comment_permissions($args);
        $this->assertTrue($perm['post']);
        $this->assertTrue($perm['view']);
    }

    public function test_comment_on_activity(): void {
        $context = context_module::instance($this->wiki->cmid);
        $this->add_comments_block("mod-wiki-*", $context->id);
        $this->setUser($this->bob);
        $context_bob = \context_user::instance($this->bob->id);
        $args = (object) ['context' => $context_bob];
        $perm = block_comments_comment_permissions($args);
        $this->assertTrue($perm['post']);
        $this->assertTrue($perm['view']);
    }
}
