<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * PHPunit tests for the cache API and in particular things in locallib.php
 *
 * This file is part of Moodle's cache API, affectionately called MUC.
 * It contains the components that are requried in order to use caching.
 *
 * @package    core
 * @category   cache
 * @copyright  2012 Sam Hemelryk
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

// Include the necessary evils.
global $CFG;
require_once($CFG->dirroot.'/cache/locallib.php');
require_once($CFG->dirroot.'/cache/tests/fixtures/lib.php');

/**
 * PHPunit tests for the cache API and in particular the cache config writer.
 *
 * @copyright  2012 Sam Hemelryk
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class core_cache_config_writer_test extends \core_phpunit\testcase {

    /**
     * Set things back to the default before each test.
     */
    public function setUp(): void {
        parent::setUp();
        cache_factory::instance(true);
        cache_factory::reset();
        cache_config_testing::create_default_configuration();
    }

    /**
     * Final task is to reset the cache system
     */
    public static function tearDownAfterClass(): void {
        cache_factory::reset();
        parent::tearDownAfterClass();
    }

    /**
     * Test getting an instance. Pretty basic.
     */
    public function test_instance() {
        $config = cache_config_writer::instance();
        $this->assertInstanceOf('cache_config_writer', $config);
    }

    /**
     * Test the default configuration.
     */
    public function test_default_configuration() {
        $config = cache_config_writer::instance();

        // First check stores.
        $stores = $config->get_all_stores();
        $hasapplication = false;
        $hassession = false;
        $hasrequest = false;
        foreach ($stores as $store) {
            // Check the required keys.
            $this->assertArrayHasKey('name', $store);
            $this->assertArrayHasKey('plugin', $store);
            $this->assertArrayHasKey('modes', $store);
            $this->assertArrayHasKey('default', $store);
            // Check the mode, we need at least one default store of each mode.
            if (!empty($store['default'])) {
                if ($store['modes'] & cache_store::MODE_APPLICATION) {
                    $hasapplication = true;
                }
                if ($store['modes'] & cache_store::MODE_SESSION) {
                    $hassession = true;
                }
                if ($store['modes'] & cache_store::MODE_REQUEST) {
                    $hasrequest = true;
                }
            }
        }
        $this->assertTrue($hasapplication, 'There is no default application cache store.');
        $this->assertTrue($hassession, 'There is no default session cache store.');
        $this->assertTrue($hasrequest, 'There is no default request cache store.');

        // Next check the definitions.
        $definitions = $config->get_definitions();
        $eventinvalidation = false;
        foreach ($definitions as $definition) {
            // Check the required keys.
            $this->assertArrayHasKey('mode', $definition);
            $this->assertArrayHasKey('component', $definition);
            $this->assertArrayHasKey('area', $definition);
            if ($definition['component'] === 'core' && $definition['area'] === 'eventinvalidation') {
                $eventinvalidation = true;
            }
        }
        $this->assertTrue($eventinvalidation, 'Missing the event invalidation definition.');

        // Next mode mappings
        $mappings = $config->get_mode_mappings();
        $hasapplication = false;
        $hassession = false;
        $hasrequest = false;
        foreach ($mappings as $mode) {
            // Check the required keys.
            $this->assertArrayHasKey('mode', $mode);
            $this->assertArrayHasKey('store', $mode);

            if ($mode['mode'] === cache_store::MODE_APPLICATION) {
                $hasapplication = true;
            }
            if ($mode['mode'] === cache_store::MODE_SESSION) {
                $hassession = true;
            }
            if ($mode['mode'] === cache_store::MODE_REQUEST) {
                $hasrequest = true;
            }
        }
        $this->assertTrue($hasapplication, 'There is no mapping for the application mode.');
        $this->assertTrue($hassession, 'There is no mapping for the session mode.');
        $this->assertTrue($hasrequest, 'There is no mapping for the request mode.');

        // Finally check config locks
        $locks = $config->get_locks();
        foreach ($locks as $lock) {
            $this->assertArrayHasKey('name', $lock);
            $this->assertArrayHasKey('type', $lock);
            $this->assertArrayHasKey('default', $lock);
        }
        // There has to be at least the default lock.
        $this->assertTrue(count($locks) > 0);
    }

    /**
     * Test updating the definitions.
     */
    public function test_update_definitions() {
        $config = cache_config_writer::instance();
        // Remove the definition.
        $config->phpunit_remove_definition('core/string');
        $definitions = $config->get_definitions();
        // Check it is gone.
        $this->assertFalse(array_key_exists('core/string', $definitions));
        // Update definitions. This should re-add it.
        cache_config_writer::update_definitions();
        $definitions = $config->get_definitions();
        // Check it is back again.
        $this->assertTrue(array_key_exists('core/string', $definitions));
    }

    /**
     * Test adding/editing/deleting store instances.
     */
    public function test_add_edit_delete_plugin_instance() {
        $config = cache_config_writer::instance();
        $this->assertArrayNotHasKey('addplugintest', $config->get_all_stores());
        $this->assertArrayNotHasKey('addplugintestwlock', $config->get_all_stores());
        // Add a default file instance.
        $config->add_store_instance('addplugintest', 'file');

        cache_factory::reset();
        $config = cache_config_writer::instance();
        $this->assertArrayHasKey('addplugintest', $config->get_all_stores());

        // Add a store with a lock described.
        $config->add_store_instance('addplugintestwlock', 'file', array('lock' => 'default_file_lock'));
        $this->assertArrayHasKey('addplugintestwlock', $config->get_all_stores());

        $config->delete_store_instance('addplugintest');
        $this->assertArrayNotHasKey('addplugintest', $config->get_all_stores());
        $this->assertArrayHasKey('addplugintestwlock', $config->get_all_stores());

        $config->delete_store_instance('addplugintestwlock');
        $this->assertArrayNotHasKey('addplugintest', $config->get_all_stores());
        $this->assertArrayNotHasKey('addplugintestwlock', $config->get_all_stores());

        // Add a default file instance.
        $config->add_store_instance('storeconfigtest', 'file', array('test' => 'a', 'one' => 'two'));
        $stores = $config->get_all_stores();
        $this->assertArrayHasKey('storeconfigtest', $stores);
        $this->assertArrayHasKey('configuration', $stores['storeconfigtest']);
        $this->assertArrayHasKey('test', $stores['storeconfigtest']['configuration']);
        $this->assertArrayHasKey('one', $stores['storeconfigtest']['configuration']);
        $this->assertEquals('a', $stores['storeconfigtest']['configuration']['test']);
        $this->assertEquals('two', $stores['storeconfigtest']['configuration']['one']);

        $config->edit_store_instance('storeconfigtest', 'file', array('test' => 'b', 'one' => 'three'));
        $stores = $config->get_all_stores();
        $this->assertArrayHasKey('storeconfigtest', $stores);
        $this->assertArrayHasKey('configuration', $stores['storeconfigtest']);
        $this->assertArrayHasKey('test', $stores['storeconfigtest']['configuration']);
        $this->assertArrayHasKey('one', $stores['storeconfigtest']['configuration']);
        $this->assertEquals('b', $stores['storeconfigtest']['configuration']['test']);
        $this->assertEquals('three', $stores['storeconfigtest']['configuration']['one']);

        $config->delete_store_instance('storeconfigtest');

        try {
            $config->delete_store_instance('default_application');
            $this->fail('Default store deleted. This should not be possible!');
        } catch (Exception $e) {
            $this->assertStringContainsString('The can not delete the default stores.', $e->getMessage());
            $this->assertInstanceOf('cache_exception', $e);
        }

        try {
            $config->delete_store_instance('some_crazy_store');
            $this->fail('You should not be able to delete a store that does not exist.');
        } catch (Exception $e) {
            $this->assertStringContainsString('The requested store does not exist.', $e->getMessage());
            $this->assertInstanceOf('cache_exception', $e);
        }

        try {
            // Try with a plugin that does not exist.
            $config->add_store_instance('storeconfigtest', 'shallowfail', array('test' => 'a', 'one' => 'two'));
            $this->fail('You should not be able to add an instance of a store that does not exist.');
        } catch (Exception $e) {
            $this->assertStringContainsString('Invalid plugin name specified. The plugin does not exist or is not valid.', $e->getMessage());
            $this->assertInstanceOf('cache_exception', $e);
        }
    }

    /**
     * Test setting some mode mappings.
     */
    public function test_set_mode_mappings() {
        $config = cache_config_writer::instance();
        $this->assertTrue($config->add_store_instance('setmodetest', 'file'));
        $this->assertTrue($config->set_mode_mappings(array(
            cache_store::MODE_APPLICATION => array('setmodetest', 'default_application'),
            cache_store::MODE_SESSION => array('default_session'),
            cache_store::MODE_REQUEST => array('default_request'),
        )));
        $mappings = $config->get_mode_mappings();
        $setmodetestfound = false;
        foreach ($mappings as $mapping) {
            if ($mapping['store'] == 'setmodetest' && $mapping['mode'] == cache_store::MODE_APPLICATION) {
                $setmodetestfound = true;
            }
        }
        $this->assertTrue($setmodetestfound, 'Set mapping did not work as expected.');
    }

    /**
     * Test setting some definition mappings.
     */
    public function test_set_definition_mappings() {
        $config = cache_config_testing::instance(true);
        $config->phpunit_add_definition('phpunit/testdefinition', array(
            'mode' => cache_store::MODE_APPLICATION,
            'component' => 'phpunit',
            'area' => 'testdefinition'
        ));

        $config = cache_config_writer::instance();
        $this->assertTrue($config->add_store_instance('setdefinitiontest', 'file'));
        $this->assertIsArray($config->get_definition_by_id('phpunit/testdefinition'));
        $config->set_definition_mappings('phpunit/testdefinition', array('setdefinitiontest', 'default_application'));

        try {
            $config->set_definition_mappings('phpunit/testdefinition', array('something that does not exist'));
            $this->fail('You should not be able to set a mapping for a store that does not exist.');
        } catch (Exception $e) {
            $this->assertEquals('Coding error detected, it must be fixed by a programmer: Invalid store name passed when updating definition mappings.', $e->getMessage());
            $this->assertInstanceOf('coding_exception', $e);
        }

        try {
            $config->set_definition_mappings('something/crazy', array('setdefinitiontest'));
            $this->fail('You should not be able to set a mapping for a definition that does not exist.');
        } catch (Exception $e) {
            $this->assertEquals('Coding error detected, it must be fixed by a programmer: Invalid definition name passed when updating mappings.', $e->getMessage());
            $this->assertInstanceOf('coding_exception', $e);
        }
    }

    /**
     * Test save config is not rewriting again with same configuration
     */
    public function test_config_save(): void {
        global $CFG;
        $directory = $CFG->dataroot.'/muc';
        $config_file_path = $directory.'/config.php';

        $config = cache_config_writer::instance();
        $rm = new ReflectionMethod($config, 'generate_configuration_array');
        $rm->setAccessible(true);
        $default_configuration = $rm->invoke($config);

        $default_configuration['siteidentifier'] = 'Test site identifier';
        $default_configuration['stores'] = [];
        $default_configuration['modemappings'] = [];
        $default_configuration['definitions'] = [];
        $default_configuration['definitionmappings'] = [];
        $default_configuration['locks'] = [];

        make_writable_directory($directory, false);

        file_put_contents($config_file_path, var_export($default_configuration, true));

        $file_time1 = filemtime($config_file_path);
        $this->assertSame(var_export($default_configuration, true), file_get_contents($config_file_path));

        sleep(1);

        $config = cache_config_writer::instance();
        $rm = new ReflectionMethod($config, 'config_save');
        $rm->setAccessible(true);
        $rm->invoke($config);

        $file_time2 = filemtime($config_file_path);

        $this->assertNotSame(var_export($default_configuration, true), file_get_contents($config_file_path));
        $this->assertNotSame($file_time1, $file_time2);

        sleep(1);

        $config2 = cache_config_writer::instance();
        $rm2 = new ReflectionMethod($config2, 'config_save');
        $rm2->setAccessible(true);
        $rm2->invoke($config2);

        $file_time3 = filemtime($config_file_path);
        $this->assertSame($file_time2, $file_time3);
    }
}
