<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Observer
 *
 * @package    bi_intellidata
 * @author     IntelliBoard
 * @copyright  2023 intelliboard.net
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace bi_intellidata\observers;

use bi_intellidata\entities\custom\entity;
use bi_intellidata\helpers\TrackingHelper;
use bi_intellidata\helpers\SettingsHelper;
use bi_intellidata\helpers\EventsHelper;
use bi_intellidata\services\events_service;
use bi_intellidata\repositories\export_log_repository;

/**
 * Event observer for deletedrecords.
 */
class record_deleted {

    /**
     * Executes when any event is triggered.
     *
     * @param \core\event\base $event
     */
    public static function execute(\core\event\base $event) {

        // Process only deleted events which includes objecttable and objectid.
        if ($event->crud != EventsHelper::CRUD_DELETED || empty($event->objecttable) || empty($event->objectid)) {
            return;
        }

        if (TrackingHelper::enabled() &&
            (int)SettingsHelper::get_setting('exportdeletedrecords') == SettingsHelper::EXPORTDELETED_TRACKEVENTS) {

            $eventdata = $event->get_data();

            $exportlogrepository = new export_log_repository();

            $datatype = $exportlogrepository->get_datatype_from_event($eventdata['eventname']);

            if (!empty($datatype)) {
                self::export_event($datatype, $eventdata);
            }
        }
    }

    /**
     * Export event.
     *
     * @param $record
     * @param $eventdata
     * @param array $fields
     * @throws \core\invalid_persistent_exception
     */
    private static function export_event($datatype, $eventdata) {

        $record = new \stdClass();
        $record->id = $eventdata['objectid'];
        $record->crud = $eventdata['crud'];

        $entity = new entity($datatype, $record);
        $data = $entity->export();

        $tracking = new events_service($datatype);
        $tracking->track($data);
    }

}
