<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Library functions to facilitate the use of JavaScript in Moodle.
 *
 * @copyright 2016 Adrian Greeve <adrian@moodle.com>
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @package core
 * @category output
 */

defined('MOODLE_INTERNAL') || die();

/**
 * This requirements manager captures the appropriate html for creating a fragment to
 * be inserted elsewhere.
 *
 * @copyright 2016 Adrian Greeve <adrian@moodle.com>
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @since Moodle 3.1
 * @package core
 * @category output
 */
class fragment_requirements_manager extends page_requirements_manager {

    /**
     * Page fragment constructor.
     */
    public function __construct() {
        parent::__construct();
        // As this is a fragment the header should already be done.
        $this->headdone = true;
    }

    /**
     * Returns js code to load amd module loader, then insert inline script tags
     * that contain require() calls using RequireJS.
     *
     * @return string
     */
    protected function get_amd_footercode($initialiseamd = false) {
        // Totara: no need to use 'core/first' any more, we use bundle configuration instead now.
        return html_writer::script(implode(";\n", $this->get_entries(static::AMD_JS_CODE)));
    }


    /**
     * Generate any HTML that needs to go at the end of the page.
     *
     * @param bool $initialiseamd initialise AMD. This is probably useful only in Totara forms ajax where we don't want it.
     * @return string the HTML code to to at the end of the page.
     */
    public function get_end_code($initialiseamd = true) {
        $output = '';

        if ($initialiseamd) {
            // Call amd init functions.
            $output .= $this->get_amd_footercode();
        }

        // Add other requested modules.
        $output .= $this->get_extra_modules_code();

        // All the other linked scripts - there should be as few as possible.
        foreach ($this->get_entries(static::JS_INCLUDES_FOOTER) as $url) {
            $output .= html_writer::script('', $url);
        }

        $stringsforjs = $this->get_entries(static::STRINGS_FOR_JS);
        if (!empty($stringsforjs)) {
            $strings = [];
            foreach (array_column($stringsforjs, 'lang_string') as $str) {
                $strings[$str->get_component()][$str->get_identifier()] = $str->out();
            }
            $output .= html_writer::script('M.util.add_strings(' . json_encode($strings) . ');');
        }

        // Add variables.
        $js_init_variables_footer = $this->get_entries(static::JS_INIT_VARIABLES_FOOTER);
        if ($js_init_variables_footer) {
            $js = '';
            foreach ($js_init_variables_footer as $data) {
                list($var, $value) = $data;
                $js .= js_writer::set_variable($var, $value, true);
            }
            $output .= html_writer::script($js);
        }

        $inyuijs = $this->get_javascript_code(false);
        $ondomreadyjs = $this->get_javascript_code(true);
        // See if this is still needed when we get to the ajax page.
        $jsinit = $this->get_javascript_init_code();
        $handlersjs = $this->get_event_handler_code();

        // There is a global Y, make sure it is available in your scope.
        $js = "(function() {{$inyuijs}{$ondomreadyjs}{$jsinit}{$handlersjs}})();";

        $output .= html_writer::script($js);

        return $output;
    }
}
