<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2021 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Tatsuhiro Kirihara <tatsuhiro.kirihara@totaralearning.com>
 * @package mod_approval
 */

namespace mod_approval\model\application\action;

use coding_exception;
use core\orm\query\builder;
use lang_string;
use mod_approval\interactor\application_interactor;
use mod_approval\model\application\activity\level_rejected;
use mod_approval\model\application\application;
use mod_approval\model\application\application_action;
use mod_approval\model\application\application_activity;
use mod_approval\model\application\application_submission;
use mod_approval\model\workflow\interaction\transition\previous;
use mod_approval\model\workflow\interaction\transition\transition_base;
use mod_approval\model\workflow\stage_type\approvals;

/**
 * 0: Reject
 */
final class reject extends action {
    public static function get_code(): int {
        return 0;
    }

    public static function get_enum(): string {
        return 'REJECT';
    }

    public static function get_label(): lang_string {
        return new lang_string('model_application_action_status_rejected', 'mod_approval');
    }

    public static function is_actionable(application_interactor $interactor): bool {
        return $interactor->can_approve();
    }

    /**
     * Reject the application.
     *
     * Does the following:
     * - creates a "reject" action record
     * - creates a "level rejected" activity record
     * - moves the application backwards, recording activities
     *
     * @param application $application
     * @param int $actor_id
     */
    public static function execute(application $application, int $actor_id): void {
        $current_state = $application->current_state;

        if (!$current_state->is_stage_type(approvals::get_code())) {
            throw new coding_exception('Cannot reject application because the state is not in approvals');
        }

        builder::get_db()->transaction(function () use ($application, $current_state, $actor_id) {
            // Mark all existing actions and submissions as superseded.
            application_action::supercede_actions_for_stage($application, $current_state->get_stage());

            // Record the action.
            application_action::create(
                $application,
                $actor_id,
                new self()
            );

            // Record the activity.
            application_activity::create(
                $application,
                $actor_id,
                level_rejected::class
            );

            // Perform a standard state change.
            static::standard_state_change($application, $current_state, $actor_id);
        });
    }

    public static function get_default_transition(): transition_base {
        return new previous();
    }
}
