<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2010 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Maria Torres <maria.torres@totaralms.com>
 * @package totara
 * @subpackage cohort
 */

defined('MOODLE_INTERNAL') || die();

global $CFG;

require_once($CFG->dirroot . '/totara/reportbuilder/tests/reportcache_advanced_testcase.php');
require_once($CFG->dirroot . '/totara/cohort/lib.php');

/**
 * Test organisation rules.
 *
 * To test, run this from the command line from the $CFG->dirroot
 * vendor/bin/phpunit totara_cohort_organisation_rules_testcase
 *
 */
class totara_cohort_organisation_rules_test extends \core_phpunit\testcase {
    private $users = null;
    private $org1 = null;
    private $org2 = null;
    private $org3 = null;
    private $org4 = null;
    private $org5 = null;
    private $orgfw = null;
    private $cohort = null;
    private $ruleset = 0;
    private $usersorg1 = array();
    private $usersorg2 = array();
    private $usersorg3 = array();
    private $usersorg4 = array();
    private $cohort_generator = null;
    private $hierarchy_generator = null;
    const TEST_ORGANISATION_COUNT_MEMBERS = 23;

    protected function tearDown(): void {
        $this->users = null;
        $this->org1 = null;
        $this->org2 = null;
        $this->org3 = null;
        $this->org4 = null;
        $this->org5 = null;
        $this->orgfw = null;
        $this->cohort = null;
        $this->ruleset = null;
        $this->usersorg1 = null;
        $this->usersorg2 = null;
        $this->usersorg3 = null;
        $this->usersorg4 = null;
        $this->cohort_generator = null;
        $this->hierarchy_generator = null;
        parent::tearDown();
    }

    /**
     * Users per organisation:
     *
     * org1 ----> user3, user6, user9, user12, user15, user18, user21.
     *
     * org2 ----> user2, user4, user8, user10, user14, user16, user20, user22.
     *
     * org3 ----> user1, user5, user7, user11, user13, user17, user19, user23.
     */
    public function setUp(): void {
        global $DB;

        parent::setup();
        $this->setAdminUser();

        $this->users = [];

        // Set totara_cohort generator.
        $this->cohort_generator = \totara_cohort\testing\generator::instance();

        // Set totara_hierarchy generator.
        $this->hierarchy_generator = \totara_hierarchy\testing\generator::instance();

        // Create organisations and organisation fw.
        $name = $this->hierarchy_generator::DEFAULT_NAME_FRAMEWORK_ORGANISATION;
        $name .= ' ' . totara_generator_util::get_next_record_number('org_framework', 'fullname', $name);
        $data = array('fullname' => $name);
        $this->orgfw = $this->hierarchy_generator->create_framework('organisation', $data);

        // Create organisations and organisation hierarchies.
        $this->assertEquals(0, $DB->count_records('org'));
        $this->org1 = $this->hierarchy_generator->create_hierarchy($this->orgfw->id, 'organisation', array('idnumber' => 'org1', 'fullname' => 'orgname1'));
        $this->org2 = $this->hierarchy_generator->create_hierarchy($this->orgfw->id, 'organisation', array('idnumber' => 'org2', 'fullname' => 'orgname2'));
        $this->org3 = $this->hierarchy_generator->create_hierarchy($this->orgfw->id, 'organisation', array('idnumber' => 'org3', 'fullname' => 'orgname3'));
        $this->assertEquals(3, $DB->count_records('org'));

        // Create some test users and assign them to an organisation.
        $this->assertEquals(2, $DB->count_records('user'));

        for ($i = 1; $i <= self::TEST_ORGANISATION_COUNT_MEMBERS; $i++) {
            $this->users[$i] = $this->getDataGenerator()->create_user();
            if ($i%3 === 0) {
                $orgid = $this->org1->id; // 7 users.
                $org = 'org1';
            } else if ($i%2 === 0){
                $orgid = $this->org2->id; // 8 users.
                $org = 'org2';
            } else {
                $orgid = $this->org3->id; // 8 users.
                $org = 'org3';
            }
            \totara_job\job_assignment::create_default($this->users[$i]->id, array('organisationid' => $orgid));
            array_push($this->{'users'.$org}, $this->users[$i]->id);
        }

        // Verify the users were created. It should match TEST_ORGANISATION_COUNT_MEMBERS + 2 users(admin + guest).
        $this->assertEquals(self::TEST_ORGANISATION_COUNT_MEMBERS + 2, $DB->count_records('user'));

        $this->usersorg1 = array_flip($this->usersorg1);
        $this->usersorg2 = array_flip($this->usersorg2);
        $this->usersorg3 = array_flip($this->usersorg3);

        // Check that organisations were assigned correctly.
        $this->assertEquals(7, $DB->count_records('job_assignment', array('organisationid' => $this->org1->id)));
        $this->assertEquals(8, $DB->count_records('job_assignment', array('organisationid' => $this->org2->id)));
        $this->assertEquals(8, $DB->count_records('job_assignment', array('organisationid' => $this->org3->id)));

        // Creating dynamic cohort and check that there are no members in the new cohort.
        $this->cohort = $this->cohort_generator->create_cohort(array('cohorttype' => cohort::TYPE_DYNAMIC));
        $this->assertTrue($DB->record_exists('cohort', array('id' => $this->cohort->id)));
        $this->assertEquals(0, $DB->count_records('cohort_members', array('cohortid' => $this->cohort->id)));

        // Create ruleset.
        $this->ruleset = cohort_rule_create_ruleset($this->cohort->draftcollectionid);
    }

    public function test_organisation_idnumber_rule() {
        global $DB;
        $this->setAdminUser();

        // Add a rule that matches users for the organisation org1. It should match 7 users.
        $this->cohort_generator->create_cohort_rule_params($this->ruleset, 'alljobassign', 'orgidnumbers', array('equal' => COHORT_RULES_OP_IN_ISEQUALTO),  array('org1'));
        cohort_rules_approve_changes($this->cohort);
        $members = $DB->get_fieldset_select('cohort_members', 'userid', 'cohortid = ?', array($this->cohort->id));
        $this->assertEquals(7, count($members));
        $this->assertEmpty(array_diff_key(array_flip($members), $this->usersorg1));
    }

    public function test_organisation_type_rule() {
        global $DB, $USER;
        $this->setAdminUser();

        // Create type of organisation.
        $newtype = new stdClass();
        $newtype->shortname = 'type1';
        $newtype->fullname = 'type1';
        $newtype->idnumber = 'typeID1';
        $newtype->description = '';
        $newtype->timecreated = time();
        $newtype->usermodified = $USER->id;
        $newtype->timemodified = time();
        $orgtype1 = $DB->insert_record('org_type', $newtype);

        // Verify the record was created correctly.
        $this->assertIsInt($orgtype1);

        // Assign the type organisation to org1.
        $this->assertTrue($DB->set_field('org', 'typeid', $orgtype1, array('id' => $this->org1->id)));

        // Create a rule that matches users in the previous created type.
        $this->cohort_generator->create_cohort_rule_params($this->ruleset, 'alljobassign', 'orgtypes', array('equal' => COHORT_RULES_OP_IN_EQUAL), array($orgtype1));
        cohort_rules_approve_changes($this->cohort);

        // It should match 7 users (org1).
        $members = $DB->get_fieldset_select('cohort_members', 'userid', 'cohortid = ?', array($this->cohort->id));
        $this->assertEquals(7, count($members));
        $this->assertEmpty(array_diff_key(array_flip($members), $this->usersorg1));
    }

    /**
     * Data provider for organisation rule.
     */
    public static function data_organisation_hierarchy() {
        $data = array(
            array(array('equal' => 1, 'includechildren' => 1),  array('org1'), 10, array('org1', 'org4')),
            array(array('equal' => 1, 'includechildren' => 0),  array('org1'), 7, array('org1')),
            array(array('equal' => 0, 'includechildren' => 1),  array('org1', 'org2'), 8, array('org3')),
        );
        return $data;
    }

   /**
    * Test organisation rule.
    *
    *  Hierarchy of organisations with their users assigned:
    *
    * org1
    *   |----> user3, user6, user9, user12, user15, user18, user21.
    *   |----> org4
    *            |----> newuser1, newuser3, newuser5.
    *
    * org2
    *   |----> user2, user4, user8, user10, user14, user16, user20, user22.
    *   |----> org5
    *            |----> newuser2, newuser4.
    *
    * @dataProvider data_organisation_hierarchy
    */
    public function test_organisation_rule($params, $organisations, $usercount, $membersmatched) {
        global $DB;
        $this->setAdminUser();

        // Create some organisatiosn to make a hierarchy.
        $data = array('idnumber' => 'og4', 'parentid' => $this->org1->id, 'fullname' => 'org4');
        $this->org4 = $this->hierarchy_generator->create_hierarchy($this->orgfw->id, 'organisation', $data);
        $data = array('idnumber' => 'org5', 'parentid' => $this->org2->id, 'fullname' => 'org5');
        $this->org5 = $this->hierarchy_generator->create_hierarchy($this->orgfw->id, 'organisation', $data);

        // Process organisations.
        $listofvalues = array();
        foreach ($organisations as $org) {
            $listofvalues[] = $this->{$org}->id;
        }

        // Create some users and assign them to the new organisations.
        $newuser1 = $this->getDataGenerator()->create_user(array('username' => 'newuser1'));
        $newuser2 = $this->getDataGenerator()->create_user(array('username' => 'newuser2'));
        $newuser3 = $this->getDataGenerator()->create_user(array('username' => 'newuser3'));
        $newuser4 = $this->getDataGenerator()->create_user(array('username' => 'newuser4'));
        $newuser5 = $this->getDataGenerator()->create_user(array('username' => 'newuser5'));

        // Assign organisations.
        \totara_job\job_assignment::create_default($newuser1->id, array('organisationid' => $this->org4->id));
        \totara_job\job_assignment::create_default($newuser2->id, array('organisationid' => $this->org5->id));
        \totara_job\job_assignment::create_default($newuser3->id, array('organisationid' => $this->org4->id));
        \totara_job\job_assignment::create_default($newuser4->id, array('organisationid' => $this->org5->id));
        \totara_job\job_assignment::create_default($newuser5->id, array('organisationid' => $this->org4->id));
        $this->usersorg4 = array_flip(array($newuser1->id, $newuser3->id, $newuser5->id));

        $this->assertEquals(3, $DB->count_records('job_assignment', array('organisationid' => $this->org4->id)));
        $this->assertEquals(2, $DB->count_records('job_assignment', array('organisationid' => $this->org5->id)));

        // Process list of users that should match the data.
        $membersincohort = array();
        foreach ($membersmatched as $member) {
            $membersincohort = $membersincohort + $this->{'users'.$member};
        }

        // Exclude admin user from this cohort.
        $this->cohort_generator->create_cohort_rule_params($this->ruleset, 'user', 'username', array('equal' => COHORT_RULES_OP_IN_NOTEQUALTO), array('admin'));

        // Create organisation rule.
        $this->cohort_generator->create_cohort_rule_params($this->ruleset, 'alljobassign', 'organisations', $params, $listofvalues);
        cohort_rules_approve_changes($this->cohort);

        // It should match:
        // 1. data1: 10 users: 7 from org1 and 3 from org4 which is a children of org1.
        // 2. data2: 7 users: users from org1 without include its children.
        // 3. data3: 8 users: users from org3.
        $members = $DB->get_fieldset_select('cohort_members', 'userid', 'cohortid = ?', array($this->cohort->id));
        $this->assertEquals($usercount, count($members));
        $this->assertEmpty(array_diff_key(array_flip($members), $membersincohort));
    }
}
