<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2023 onwards Totara Learning Solutions LTDvs
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Murali Nair <murali.nair@totaralearning.com>
 * @package totara_competency
 * @category test
 */

 use core\collection;
 use core\entity\user;
 use totara_competency\entity\assignment;
 use totara_competency\entity\competency_achievement;
 use totara_competency\entity\competency_assignment_user;
 use totara_competency\user_groups;
 use totara_hierarchy\entity\scale_value;

 require_once(__DIR__.'/totara_competency_testcase.php');

 /**
  * @group totara_competency
  */
abstract class totara_competency_perform_overview_testcase extends totara_competency_testcase {
    /**
     * Checks if the results match the expected ones.
     *
     * @param stdClass[] $raw_expected expected achievements. Each item needs to
     *        have a specific set of fields - see self::format_raw_expected().
     * @param collection<competency_achievement> $results achievements meeting a
     *        given user/overview state combination.
     * @param stdClass $test_data data as generated by self::create_test_data();
     *        used to look up generated entity ids given a name, idnumber, etc.
     */
    protected function assert_overview(
        array $raw_expected,
        collection $results,
        stdClass $test_data
    ): void {
        $expected = $this->format_raw_expected($raw_expected, $test_data)
            ->key_by('key');

        $actual = $this->format_raw_actual($results)->key_by('key');

        foreach ($actual as $key => $details) {
            $stringified = print_r($details, true);

            $exp = $expected[$key] ?? null;
            self::assertNotNull($exp, "Should not be picked up: $stringified");
            self::assertEquals($details, $exp, 'wrong details');

            unset($expected[$key]);
        }

        $remaining = $expected
            ->map(fn(stdClass $exp): string => print_r($exp, true))
            ->all();

        if ($remaining) {
            // If it gets to here, it means the data provider did not pick up
            // all the expected achievements.
            self::fail(
                'Achievements were not picked up: ' . implode('\n', $remaining)
            );
        }
    }

    /**
     * Creates test data.
     *
     * Note: DO NOT INVOKE totara_competency\expand_task after using this method;
     * it will create extra records and mess everything up.
     *
     * @param stdClass[] $raw_achievements list of competency achievements to
     *        create. Each stdClass item is expected to have these fields:
     *        - string comp_name: name of competency to create.
     *        - mixed[] user_group: [string type, string idnumber] indicating a
     *          user group to create and assign to the competency. The type is
     *          one of the constants in the user_groups class.
     *        - times => list of [days ago, rating] tuples indicating records to
     *          create.
     * @param array<mixed[]> $raw_scale indicates the scale values to create. See
     *        self::create_competency_framework() method header for how this is
     *        interpreted.
     * @param int $now the 'current' time in seconds since the Epoch.
     * @param $lowest_scale_value_is_not_started sets the lowest is not started
     *        value for the created competency scale.
     *
     * @return stdClass test data with the following fields:
     *         - collection<string,collection<competency_achievements>> achievements
     *           mapping of achievement keys (from self::achievement_key()) to
     *           the achievements for that user/competency/assignment group
     *           combination. The achievements are sorted in ascending time
     *           order ie earliest first.
     *         - collection<string,int> competencies: mapping of competency names
     *           to the corresponding competency ids.
     *         - int $now the 'current' time in seconds since the Epoch.
     *         - collection<string,int> ratings: mapping of rating names to the
     *           the corresponding rating value ids.
     *         - collection<string,int> user_groups: mapping of *idnumbers* to
     *           the user group id.
     *         - user $user the user for whom the overview is being generated.
     */
    protected function create_test_data(
        array $raw_achievements,
        array $raw_scale,
        int $now,
        bool $lowest_scale_value_is_not_started = false
    ): stdClass {
        $user = $this->create_user();
        [$fw, $ratings] = $this->create_competency_framework(
            $raw_scale, $lowest_scale_value_is_not_started
        );

        $generator = $this->generator();
        $assign_generator = $generator->assignment_generator();

        $achievements = collection::new([]);
        $competencies = collection::new([]);
        $user_groups = collection::new([]);

        foreach ($raw_achievements as $raw) {
            $comp_name = $raw->comp_name;
            $comp_id = $competencies->item($comp_name);

            if (is_null($comp_id)) {
                $comp_id = $generator->create_competency($comp_name, $fw)->id;
                $competencies->set($comp_id, $comp_name);
            }

            [$user_group_type, $user_group_idn] = $raw->user_group;
            $user_group_id = $user_groups->item($user_group_idn);

            if (is_null($user_group_id)) {
                $attrs = ['idnumber' => $user_group_idn];

                if ($user_group_type === user_groups::USER) {
                    $user_group_id = $this->create_user($attrs)->id;
                } else {
                    $grp_fn = sprintf('create_%s_and_add_members', $user_group_type);
                    $user_group_id = $assign_generator->$grp_fn($user, $attrs)->id;
                }

                $user_groups = $user_groups->set($user_group_id, $user_group_idn);
            }

            $assignment = null;
            if ($user_group_type === user_groups::USER) {
                $assignment = $assign_generator->create_user_assignment(
                    $comp_id, $user_group_id
                );
            } else {
                $assign_fn = sprintf('create_%s_assignment', $user_group_type);
                $assignment = $assign_generator->$assign_fn($comp_id, $user_group_id);
            }

            $achievements_by_group = $this->create_achievements(
                $raw->times, $user, new assignment($assignment), $ratings, $now
            );

            $achievement_key = $this->achievement_key(
                $user->id, $comp_id, $user_group_id, $user_group_type
            );

            $achievements->set($achievements_by_group, $achievement_key);
        }

        return (object) [
            'achievements' => $achievements,
            'competencies' => $competencies,
            'now' => $now,
            'ratings' => $ratings,
            'user_groups' => $user_groups,
            'user' => $user
        ];
    }

    /**
     * Creates a key for an achievement. An achievement is essentially unique
     * for a user/competency/assignment combination.
     *
     * @param int $user_id associated user id.
     * @param int $comp_id associated competency id.
     * @param int $user_group_id associated user group id.
     * @param string $user_group_type one of the user_groups class constants.
     *
     * @return string the key.
     */
    protected function achievement_key(
        int $user_id,
        int $comp_id,
        int $user_group_id,
        string $user_group_type
    ): string {
        return sprintf(
            'user=%d:comp=%d|user_group=%d|user_group_type=%s',
            $user_id,
            $comp_id,
            $user_group_id,
            $user_group_type
        );
    }

    /**
     * Creates a competency achievement records for the specified user/assignment
     * combination.
     *
     * @param array<mixed[]> $times list of [days ago, rating] tuples indicating
     *        the timestamps for the achievement records to be created.
     * @param user $user user for whom the achievement records are being created.
     * @param assignment $assignment assignment for whom the achievement records
     *        are being created.
     * @param collection<string,scale_value> $ratings mapping of rating names to
     *        values.
     * @param int $now the 'current' time in seconds since the Epoch.
     *
     * @return collection<competency_achievement> created achievements sorted
     *         in ascending updated time order ie earliest first.
     */
    protected function create_achievements(
        array $times,
        user $user,
        assignment $assignment,
        collection $ratings,
        int $now
    ): collection {
        $user_id = $user->id;
        $competency_id = $assignment->competency_id;
        $assignment_id = $assignment->id;

        $common = [
            'competency_id' => $competency_id,
            'assignment_id' => $assignment_id,
            'user_id' => $user_id,
            'status' => competency_achievement::SUPERSEDED
        ];

        $achievements = collection::new($times)
            ->map(
                function (array $tuple) use ($ratings, $now): array {
                    [$days_ago, $rating] = $tuple;

                    $value = $ratings->item($rating);
                    $instant = $this->days_ago_timestamp($now, $days_ago);

                    return [
                        'scale_value_id' => $value ? $value->id : null,
                        'proficient' => $value ? $value->proficient : false,
                        'time_created' => $instant,
                        'time_status' => $instant,
                        'time_proficient' => $instant,
                        'time_scale_value' => $instant,
                        'last_aggregated' => $instant,
                    ];
                }
            )
            ->map(
                function (array $attrs) use ($common): competency_achievement {
                    $values = array_merge($attrs, $common);

                    // Cannot use this->create_achievement_record() because that
                    // method cannot create null (ie not started) achievements.
                    return (new competency_achievement($values))->save();
                }
            )
            ->sort(
                function (competency_achievement $a, competency_achievement $b): int {
                    return $a->time_scale_value <=> $b->time_scale_value;
                }
            );

        $latest = $achievements->last();
        $latest->status = competency_achievement::ACTIVE_ASSIGNMENT;
        $latest->save();

        // Must maintain the illusion that the competency/group assignment was
        // done before the first achievement was ever recorded.
        $user_group_assignment_time = $latest->time_created - (2 * WEEKSECS);
        $assignment->created_at = $user_group_assignment_time;
        $assignment->updated_at = $user_group_assignment_time;
        $assignment->save();

        // And also the user assignment was done before the first achievement.
        // Note: Do not run the expand_task after this, it create extra records
        // and messes things up.
        $user_assignment_time = $user_group_assignment_time + WEEKSECS;
        $user_assignment_repo = competency_assignment_user::repository()
            ->where('user_id', $user_id)
            ->where('competency_id', $competency_id)
            ->where('assignment_id', $assignment_id);

        $user_assignment = $user_assignment_repo->one() ?? new competency_assignment_user();

        $user_assignment->user_id = $user_id;
        $user_assignment->competency_id = $competency_id;
        $user_assignment->assignment_id = $assignment_id;
        $user_assignment->created_at = $user_assignment_time;
        $user_assignment->updated_at = $user_assignment_time;

        $user_assignment->save();

        return $achievements;
    }

    /**
     * Creates a test competency framework.
     *
     * @param array<mixed[]> $scale_values [string name, bool proficient] tuples
     *        indicating scale values to create. This list must be ordered _from
     *        the lowest rating to the highest_. Also if a tuple other than the
     *        highest has a true value for proficient, then all the scale values
     *        from that one to the highest one are marked as proficient.
     * @param $lowest_scale_value_is_not_started sets the lowest is not started
     *        value for the created competency scale.
     *
     * @return mixed[] [competency_framework framework,  collection<string,
     *         scale_value> ratings by name to rating] tuple.
     */
    protected function create_competency_framework(
        array $scale_values, bool $lowest_scale_value_is_not_started
    ): array {
        $values = [];
        $mark_proficient = false;
        $sort_order = count($scale_values);

        // For competency scales, the lowest sorted entry is the most complete
        // while the highest is the least complete. However, for these overview
        // tests, the caller passes in scale values _from least complete to most
        // complete_. Hence the decreasing sort order below.
        foreach ($scale_values as $i => $tuple) {
            [$value_name, $proficient] = $tuple;
            $mark_proficient = $mark_proficient ?: $proficient;

            $values[] = [
                'name' => $value_name,
                'proficient' => $mark_proficient,
                'sortorder' => $sort_order--,
                'default' => $i === 0
            ];
        }

        $generator = $this->generator();
        $scale = $generator->create_scale('my_test_scale', null, $values);

        $scale->lowest_is_not_started = $lowest_scale_value_is_not_started;
        $scale->save();

        $fw = $generator->create_framework($scale);
        $ratings = $scale->values->key_by('name');

        return [$fw, $ratings];
    }

    /**
     * Calculates a timestamp for the start of day, X days ago.
     *
     * @param int $now the 'current' time.
     * @param int $days_ago the start of the overview period.
     *
     * @return int the timestamp.
     */
    protected function days_ago_timestamp(int $now, int $days_ago): int {
        return $now - ($days_ago * DAYSECS);
    }

    /**
     * Takes retrieved achievements and converts them into a format that can be
     * compared with the expected overview achievements.
     *
     * @param collection<competency_achievement> $actual retrieved achievements.
     *
     * @return collection<stdClass> list of items with these fields:
     *        - string key: unique user id/competency id/user group idnumber/
     *          user group type combination identifying this achievement
     *        - string comp_name: competency name.
     *        - string user_group: user group idnumber.
     *        - ?int rating: rating value id or null if not started.
     *        - ?string rating_name: rating name or null if not started.
     *        - string time: 'Y-m-d H:i:s' formatted time of achievement.
     */
    protected function format_raw_actual(collection $actual): collection {
        $fmt = function (int $timestamp): string {
            return (new DateTimeImmutable())
                ->setTimestamp($timestamp)
                ->format('Y-m-d H:i:s');
        };

        return $actual->map(
            function (competency_achievement $achievement) use ($fmt): stdClass {
                $assignment = $achievement->assignment;
                $scale_value_id = (int)$achievement->scale_value_id;

                $uid = $achievement->user_id;
                $competency_id = $achievement->competency_id;

                return (object) [
                    'key' => $this->achievement_key(
                        $uid,
                        $competency_id,
                        $assignment->user_group_id,
                        $assignment->user_group_type
                    ),
                    'comp_name' => $achievement->competency->fullname,
                    'user_group' => $assignment->user_group->idnumber,
                    'rating' => $scale_value_id ? $scale_value_id : null,
                    'rating_name' => $scale_value_id
                        ? $achievement->value->name
                        : null,
                    'time' => $fmt($achievement->time_scale_value)
                ];
            }
        );
    }

    /**
     * Takes raw expected achievements and converts them into a format that can
     * be compared with the actual overview achievements.
     *
     * @param stdClass[] $raw expected achievements. Each item in this list has
     *        these fields:
     *        - string comp_name: competency name
     *        - array user_group: [string group type, string group idnumber]
     *          tuple. The group type is a constants from the user_groups class.
     *        - int time: the achievement time
     *        - string rating: the achievement scale value
     * @param stdClass $test_data data as generated by self::create_test_data();
     *        used to look up generated entity ids given a name, idnumber, etc.
     *
     * @return collection<stdClass> list of formatted items in the same format
     *         as those returned from self::format_raw_actual().
     */
    protected function format_raw_expected(
        array $raw, stdClass $test_data
    ): collection {
        $fmt = function (int $timestamp): string {
            return (new DateTimeImmutable())
                ->setTimestamp($timestamp)
                ->format('Y-m-d H:i:s');
        };

        return collection::new($raw)->map(
            function (stdClass $exp) use ($test_data, $fmt): stdClass {
                $uid = $test_data->user->id;
                $competency_id = $test_data->competencies->item($exp->comp_name);

                [$user_group_type, $user_group_idn] = $exp->user_group;
                $value = $test_data->ratings->item($exp->rating);
                $timestamp = $this->days_ago_timestamp(
                    $test_data->now, $exp->time
                );

                $key = $this->achievement_key(
                    $uid,
                    $competency_id,
                    $test_data->user_groups->item($user_group_idn),
                    $user_group_type
                );

                return (object) [
                    'key' => $key,
                    'comp_name' => $exp->comp_name,
                    'user_group' => $user_group_idn,
                    'rating' => $value ? (int)$value->id : null,
                    'rating_name' => $exp->rating,
                    'time' => $fmt($timestamp)
                ];
            }
        );
    }
}