<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2024 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Navjeet Singh <navjeet.singh@totara.com>
 * @package totara_hierarchy
 */

namespace totara_hierarchy\data_providers;

use coding_exception;
use core\orm\pagination\cursor_paginator;
use core\orm\pagination\offset_cursor_paginator;
use core\pagination\base_cursor;
use core\pagination\cursor;
use core\pagination\offset_cursor;
use totara_hierarchy\entity\filters\base_filters;
use totara_hierarchy\entity\hierarchy_item;

/**
 * Base Data provider for fetching hierarchy prefix.
 */
abstract class base {
    public const DEFAULT_PAGE_SIZE = 20;

    private const VALID_ORDER_BY_FIELDS = ['id', 'fullname', 'sortthread'];
    private const VALID_ORDER_DIRECTION = ['asc', 'desc'];

    protected $page_size = self::DEFAULT_PAGE_SIZE;
    protected $order_by = 'fullname';
    protected $order_direction = 'asc';
    protected $filters = [];

    /**
     * Get hierarchy entity class
     * @return string|hierarchy_item
     */
    abstract protected function get_entity_class(): string;

    /**
     * Get hierarchy filter class
     * @return string
     */
    protected function get_filter_class(): string {
        return base_filters::class;
    }

    /**
     * Default constructor.
     */
    public function __construct() {
        $this->set_filters([]);
    }

    /**
     * Set the number of entries retrieved per page.
     *
     * @param int $page_size page size.
     *
     * @return base this object.
     */
    public function set_page_size(int $page_size): base {
        $this->page_size = $page_size > 0 ? $page_size : self::DEFAULT_PAGE_SIZE;
        return $this;
    }

    /**
     * Set the sorting parameters to use when retrieving hierarchy items.
     *
     * @param string $order_by position field on which to sort.
     * @param string $order_direction sorting order either 'ASC' or 'DESC'.
     *
     * @return base this object.
     */
    public function set_order(string $order_by = 'fullname', string $order_direction = 'asc'): base {
        $order_by = strtolower($order_by);
        if (!in_array($order_by, self::VALID_ORDER_BY_FIELDS)) {
            $allowed = implode(', ', self::VALID_ORDER_BY_FIELDS);
            throw new coding_exception("Ordering can only be by these fields: $allowed");
        }
        $this->order_by = $order_by;

        $order_direction = strtolower($order_direction);
        if (!in_array($order_direction, self::VALID_ORDER_DIRECTION)) {
            $allowed = implode(', ', self::VALID_ORDER_DIRECTION);
            throw new coding_exception("Order must be one of these: $allowed");
        }
        $this->order_direction = $order_direction;

        return $this;
    }

    /**
     * Indicates the filters to use when retrieving collection.
     *
     * @param array $filters mapping of collection fields to search values.
     *
     * @return base this object.
     */
    public function set_filters(array $filters): base {
        $new_filters = [];
        foreach ($filters as $key => $value) {
            $filter_value = $this->validate_filter_value($value);
            if (is_null($filter_value)) {
                continue;
            }

            /** @var base_filters $class*/
            $class = $this->get_filter_class();

            $filter = $class::for_key($key, $filter_value);
            if (!$filter) {
                throw new coding_exception("Unknown filter: '$key'");
            }

            $filter->set_entity_class($this->get_entity_class());

            $new_filters[$key] = $filter;
        }

        $this->filters = $new_filters;
        return $this;
    }

    /**
     * Checks whether the filter value is "valid". "Valid" means:
     *   - a non empty string after it has been trimmed
     *
     * @param mixed $value the value to check.
     *
     * @return mixed the filter value if it is "valid" or null otherwise.
     */
    private function validate_filter_value($value) {
        if (is_string($value)) {
            $str_value = trim($value);
            return $str_value ? $str_value : null;
        }

        return $value;
    }

    /**
     * Returns a list of entity meeting the previously set search criteria.
     *
     * @param base_cursor $cursor indicates which "page" of collection to retrieve.
     * @param array $with indicates which "relationship" of collection to retrieve.
     * @param string $raw_query indicates raw sql query
     * @param array $group_by indicates group_by query
     * @param string $alias indicates alias of the table.
     *
     * @return hierarchy_item[] the retrieved collection entities.
     */
    public function fetch(?base_cursor $cursor = null, ?array $with = [], ?string $selected_fields = '',?array $group_by = [], ?string $alias = 'hier'): array {
        $class = $this->get_entity_class();
        $repository = $class::repository();

        $repository->as($alias);

        if (!empty($selected_fields)) {
            $repository->select_raw($selected_fields);
        }

        $repository->set_filters($this->filters)
            ->order_by($this->order_by, $this->order_direction);

        if (!empty($with)) {
            $repository->with($with);
        }

        if (!empty($group_by)) {
            $repository->group_by($group_by);
        }

        // Use offset paginator if offset cursor is provided
        if (!is_null($cursor) && $cursor instanceof offset_cursor) {
            $paginator = new offset_cursor_paginator($repository, $cursor);
        } else {
            // use the cursor paginator
            $pages = $cursor ? $cursor : cursor::create()->set_limit($this->page_size);
            $paginator = new cursor_paginator($repository, $pages, true);
        }

        return $paginator->get();
    }
}