<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2010 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Maria Torres <maria.torres@totaralms.com>
 * @author Rob Tyler <rob.tyler@totaralms.com>
 * @package totara_hierarchy
 * @subpackage test
 */

namespace totara_hierarchy\testing;

use coding_exception;
use competency;
use core\entity\user;
use core\orm\query\builder;
use goal;
use hierarchy;
use hierarchy_goal\entity\company_goal as company_goal_entity;
use hierarchy_goal\entity\company_goal_assignment;
use hierarchy_goal\entity\goal_item_history;
use hierarchy_goal\entity\personal_goal as personal_goal_entity;
use hierarchy_goal\event\personal_created;
use hierarchy_goal\personal_goal_assignment_type;
use stdClass;
use totara_generator_util;

/**
 * Hierarchy generator
 *
 * @package totara_hierarchy
 * @subpackage test
 */
final class generator extends \core\testing\component_generator {

    // Default names when created a framework.
    const DEFAULT_NAME_FRAMEWORK_COMPETENCY = 'Test Competency Framework';
    const DEFAULT_NAME_FRAMEWORK_GOAL = 'Test Goal Framework';
    const DEFAULT_NAME_FRAMEWORK_ORGANISATION = 'Test Organisation Framework';
    const DEFAULT_NAME_FRAMEWORK_POSITION = 'Test Position Framework';

    // Default names when created a hierarchy.
    const DEFAULT_NAME_HIERARCHY_COMPETENCY = 'Test Competency';
    const DEFAULT_NAME_HIERARCHY_GOAL = 'Test Goal';
    const DEFAULT_NAME_HIERARCHY_ORGANISATION = 'Test Organisation';
    const DEFAULT_NAME_HIERARCHY_POSITION = 'Test Position';

    /**
     * @var array Map of hierarchy type and prefix
     */
    private $hierarchy_type_prefix = array('competency' => 'comp',
                                           'goal'=> 'goal',
                                           'organisation' => 'org',
                                           'position' => 'pos');
    /**
     * @var array integer Number of items to be assigned.
     */
    private $hierarchy_assign_quantities = array(2, 4, 8, 16, 32, 64);

    /**
     * @var integer Keep track of how many frameworks have been created.
     */
    protected $frameworkcount = array ('competency' => 0,
                                     'goal' => 0,
                                     'organisation' => 0,
                                     'position' => 0);
    /**
     * @var integer Keep track of how many hierarchies have been created.
     */
    protected $hierarchycount = array ('competency' => 0,
                                     'goal' => 0,
                                     'organisation' => 0,
                                     'position' => 0);

    /**
     * @var int[] Keep track of how many types have been created.
     */
    protected $type_count = [
        'competency' => 0,
        'goal' => 0,
        'organisation' => 0,
        'position' => 0,
    ];

    protected function __construct() {
        global $CFG;
        parent::__construct();
        require_once($CFG->dirroot . '/totara/hierarchy/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/competency/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/position/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/competency/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/goal/lib.php');
        require_once($CFG->dirroot . '/totara/customfield/fieldlib.php');
    }

    public function reset() {
        $this->frameworkcount = [
            'competency' => 0,
            'goal' => 0,
            'organisation' => 0,
            'position' => 0,
        ];
        $this->hierarchycount = [
            'competency' => 0,
            'goal' => 0,
            'organisation' => 0,
            'position' => 0,
        ];
        $this->type_count = [
            'competency' => 0,
            'goal' => 0,
            'organisation' => 0,
            'position' => 0,
        ];
        parent::reset();
    }

    /**
     * Redirect behat generator with appropriate prefix.
     */
    public function create_pos_frame($data) {
        return $this->create_framework('position', $data);
    }

    public function create_org_frame($data) {
        return $this->create_framework('organisation', $data);
    }

    public function create_comp_frame($data) {
        return $this->create_framework('competency', $data);
    }

    public function create_goal_frame($data) {
        return $this->create_framework('goal', $data);
    }

    /**
     * Create a framework for the given prefix.
     *
     * @param string $prefix Prefix that identifies the type of hierarchy (position, organisation, etc)
     * @param array $record
     * @return stdClass hierarchy framework
     *
     * @todo Define an array of default values then use
     *       array_merge($default_values,$record) to
     *       merge in the optional record data and reduce
     *       / remove the need for multiple statements
     *       beginning with: if (!isset($record['...
     */
    public function create_framework($prefix, $record = array()) {
        global $USER, $DB, $CFG;
        require_once($CFG->dirroot . '/totara/hierarchy/lib.php');

        $record = (array) $record;
        $shortprefix = hierarchy::get_short_prefix($prefix);
        // Increment the count for the given framework.
        $i = ++$this->frameworkcount[$prefix];

        if (!isset($record['visible'])) {
            $record['visible'] = 1;
        }

        if (!isset($record['fullname'])) {
            $defaultnameconst = 'self::DEFAULT_NAME_FRAMEWORK_' . strtoupper($prefix);
            $record['fullname'] = trim(constant($defaultnameconst)) . ' ' .$i;
        }

        if (!isset($record['idnumber'])) {
            $record['idnumber'] = totara_generator_util::create_short_name($record['fullname']);
        }

        if (!isset($record['description'])) {
            $record['description'] = '<p>' . $record['fullname'] . ' description</p>';
        }

        // Get the sort order from the database.
        if (!isset($record['sortorder'])) {
            $record['sortorder'] = $DB->get_field($shortprefix.'_framework', 'MAX(sortorder) + 1', array());
        }
        // A sort order may not have been found in the database or may have an invalid 0 or NULL value.
        if (!$record['sortorder']) {
            $record['sortorder'] = 1;
        }

        if (!isset($record['hidecustomfields'])) {
            $record['hidecustomfields'] = '0';
        }

        $record['timecreated'] = time();
        $record['timemodified'] = $record['timecreated'];
        $record['usermodified'] = $USER->id;

        $framework_id = $DB->insert_record($shortprefix.'_framework', $record);
        $framework = $DB->get_record($shortprefix.'_framework', array('id' => $framework_id));

        if (isset($record['scale']) && !is_numeric($record['scale'])) {
            // Scale name has been specified, so load the scale with that name.
            $record['scale'] = $DB->get_field($shortprefix . '_scale', 'id', ['name' => $record['scale']]);
        } else if (!isset($record['scale'])) {
            $record['scale'] = 1;
        }

        // If this is an competency or organisation
        // framework we need to assign a scale to it.
        if ($prefix == 'competency' || $prefix == 'goal') {
            // We need to assign a scale to the
            $scale_assign = new stdClass();
            $scale_assign->scaleid = $record['scale'];
            $scale_assign->frameworkid = $framework_id;
            $scale_assign->timemodified = time();
            $scale_assign->usermodified = $USER->id;

            $scale_assign_id = $DB->insert_record($shortprefix . '_scale_assignments', $scale_assign);
        }

        return $framework;
    }

    /**
     * Assign some learners to a company goal individually
     *
     * @param int $goalid - the id of a company level goal
     * @param array(int) $userids - an array of userids to be assigned
     * @param int|null $timecreated
     * @return bool - whether the users were successfully assigned or not
     */
    public function goal_assign_individuals($goalid, $userids = array(), ?int $timecreated = null) {
        global $USER, $DB;

        $goalinfo = goal::goal_assignment_type_info(GOAL_ASSIGNMENT_INDIVIDUAL, $goalid);
        $field = $goalinfo->field;

        // Set up the default scale value for the goal.
        $sql = "SELECT s.defaultid
                FROM {goal} g
                JOIN {goal_scale_assignments} sa
                    ON g.frameworkid = sa.frameworkid
                JOIN {goal_scale} s
                    ON sa.scaleid = s.id
                WHERE g.id = :gid";
        $scale = $DB->get_record_sql($sql, array('gid' => $goalid));

        // There should always be a goal_scale, something is horribly wrong if there isn't.
        if (empty($scale)) {
            return false;
        }

        foreach ($userids as $uid) {

            $scale_default = new stdClass();
            $scale_default->goalid = $goalid;
            $scale_default->userid = $uid;
            $scale_default->scalevalueid = $scale->defaultid;

            // Add the individual assignment.
            $assignment = new stdClass();
            $assignment->assignmentid = 0;
            $assignment->$field = $uid;
            $assignment->assigntype = GOAL_ASSIGNMENT_INDIVIDUAL;
            $assignment->goalid = $goalid;
            $assignment->timemodified = $timecreated ?? time();
            $assignment->timecreated = $timecreated ?? time();
            $assignment->usermodified = $USER->id;
            $assignment->includechildren = 0;

            $assignment->id = $DB->insert_record($goalinfo->table, $assignment);

            $goalrecords = goal::get_goal_items(array('goalid' => $goalid, 'userid' => $uid), goal::SCOPE_COMPANY);
            if (empty($goalrecords)) {
                goal::insert_goal_item($scale_default, goal::SCOPE_COMPANY);
            }
        }
    }

    /**
     * Create a scale for the given prefix.
     *
     * We need to create the scale with a dummy default value, so we can create the values
     * with the correct scaleid, then we update the default value when we know the correct valueid.
     *
     * @param string $prefix Prefix that identifies the type of hierarchy (competency, goal)
     * @param array $scaledata - The scale item record
     * @param array $valuedata - An array of scale value items, note one value should have a default value set to 1.
     * @return stdClass scale database object
     */
    public function create_scale($prefix, $scaledata = array(), $valuedata = array()) {
        global $USER, $DB;

        // Create the scale item, filling in any missing information.
        $sdefaults = ['name' => $prefix . '_scale',
                      'description' => $prefix . '_scale',
                      'timemodified' => time(),
                      'usermodified' => $USER->id,
                      'defaultid' => 1];
        $scaledata = array_merge($sdefaults, $scaledata);
        $scaleid = $DB->insert_record("{$prefix}_scale", $scaledata);

        // Create the scale values, filling in any missing information.
        $vdefaults = ['name' => $prefix . '_scale_value',
                      'proficient' => 0,
                      'scaleid' => $scaleid,
                      'timemodified' => time(),
                      'usermodified' => $USER->id];

        // You can't have a scale without values, so if values is empty chuck in these.
        if (empty($valuedata)) {
            $valuedata = [
                1 => ['name' => 'Assigned', 'proficient' => 0, 'sortorder' => 3, 'default' => 1],
                2 => ['name' => 'Progress', 'proficient' => 0, 'sortorder' => 2, 'default' => 0],
                3 => ['name' => 'Complete', 'proficient' => 1, 'sortorder' => 1, 'default' => 0]
            ];
        }

        $value = null;
        $defaultid = null;
        foreach ($valuedata as $vdata) {
            $vdata = array_merge($vdefaults, $vdata);

            $valueid = $DB->insert_record("{$prefix}_scale_values", $vdata);

            if (!empty($vdata['default'])) {
                $defaultid = $valueid;
            }
        }

        // If a default value hasn't been specified, just use the last one.
        if (empty($defaultid)) {
            $defaultid = $valueid;
        }

        $DB->set_field("{$prefix}_scale", 'defaultid', $defaultid, ['id' => $scaleid]);

        if ($prefix == 'comp') {
            // Work out the minimum proficient value.
            $minproficiencyid = $DB->get_records(
                'comp_scale_values',
                array('scaleid' => $scaleid, 'proficient' => 1),
                'sortorder DESC',
                'id',
                0,
                1
            );
            if (count($minproficiencyid) > 0) {
                $minproficiencyid = array_shift($minproficiencyid)->id;
                $DB->set_field('comp_scale', 'minproficiencyid', $minproficiencyid, array('id' => $scaleid));
            }
        }

        return $DB->get_record("{$prefix}_scale", array('id' => $scaleid));
    }

    /**
     * Create a personal goal for a user
     *
     * @param int $userid         The id of the user to create the goal for
     * @param array $goaldata     The data for the goal, anything not provided will use default data
     *                            NOTE: Customfields can be passed through goaldata with the key 'cf_<fieldshortname>'
     * @param bool $trigger_event Whether to trigger a personal_created event or not.
     * @return stdClass           The database record for the created personal goal
     */
    public function create_personal_goal($userid, $goaldata = array(), bool $trigger_event = false) {
        global $USER, $DB;

        $now = time();
        $defaultdata = ['name' =>  'Personal Goal',
                        'targetdate' => $now + (60 * DAYSECS),
                        'assigntype' => GOAL_ASSIGNMENT_SELF,
                        'timecreated' => $now,
                        'usercreated' => $USER->id,
                        'timemodified' => $now,
                        'usermodified' => $USER->id,
                        'deleted' => 0,
                        'typeid' => null];
        $goaldata = array_merge($defaultdata, $goaldata);
        $goaldata['userid'] = $userid;

        // Originally targetdate could be null, but since TL-35631 the column is
        // non nullable. Must catch any nulls passing through here.
        $goaldata['targetdate'] = $goaldata['targetdate'] ?? 0;

        $goalid = $DB->insert_record('goal_personal', $goaldata);
        $goal = $DB->get_record('goal_personal', ['id' => $goalid]);

        if (!empty($goal->typeid)) {
            $fields = $DB->get_records('goal_user_info_field', ['typeid' => $goal->typeid]);

            // Initialize all the fields.
            foreach ($fields as $field) {

                $fieldname = 'cf_' . $field->shortname;
                if (!empty($goaldata[$fieldname]) || !empty($field->defaultdata)) {
                    $input = "customfield_{$field->datatype}{$field->typeid}";

                    $item = new \stdClass();
                    $item->id = $goal->id;
                    $item->typeid = $goal->typeid;
                    $item->{$input} = !empty($goaldata[$fieldname]) ? $goaldata[$fieldname] : $field->defaultdata;
                    customfield_save_data($item, 'goal_user', 'goal_user');
                }
            }
        }

        // Insert history record when created with a scale value.
        if (!empty($goaldata['scalevalueid'])) {
            $DB->insert_record('goal_item_history', (object)[
                'scope' => goal::SCOPE_PERSONAL,
                'itemid' => $goal->id,
                'scalevalueid' => $goaldata['scalevalueid'],
                'timemodified' => $goaldata['timecreated'] ?? time(),
                'usermodified' => $USER->id,
            ]);
        }

        if ($trigger_event) {
            personal_created::create_from_instance($goal)->trigger();
        }

        return $goal;
    }

    /**
     * Create a type for personal goals so custom fields can be added.
     *
     * @param array $typedata The data for the type, anything not provided will use default data
     * @return stdClass       The database record for the created type
     */
    public function create_personal_goal_type($typedata = array()) {
        global $USER, $DB;

        $defaultdata = ['fullname' => 'Personal Goal Type',
                        'shortname' => 'pgoaltype',
                        'idnumber' => 'pgtype123',
                        'timecreated' => time(),
                        'timemodified' => time(),
                        'usermodified' => $USER->id,
                        'audience' => 0];
        $typedata = array_merge($defaultdata, $typedata);

        $typeid = $DB->insert_record('goal_user_type', $typedata);

        return $DB->get_record('goal_user_type', ['id' => $typeid]);
    }

    /**
     * Stub function to call create_personal_goal_type_customfield() with the correct
     * variables to create a menu type custom field
     *
     * @param array $data - The basic data to create the customfield with
     * @return void
     */
    public function create_personal_goal_type_menu($data) {
        $customfield = $data;
        $customfield['param1'] = "1234"."\n"."2345"."\n"."3456"."\n"."4567";
        $this->create_personal_goal_type_customfield('menu', $customfield);
    }

    /**
     * Stub function to call create_personal_goal_type_customfield() with the correct
     * variables to create a text type custom field
     *
     * @param array $data - The basic data to create the customfield with
     * @return void
     */
    public function create_personal_goal_type_text($data) {
        $customfield = $data;
        $customfield['param1'] = 30;
        $customfield['param2'] = 2048;
        $this->create_personal_goal_type_customfield('text', $customfield);
    }

    /**
     * Stub function to call create_personal_goal_type_customfield() with the correct
     * variables to create a datetime type custom field
     *
     * @param array $data - The basic data to create the customfield with
     * @return void
     */
    public function create_personal_goal_type_datetime($data) {
        $customfield = $data;
        $customfield['param1'] = date("Y")-1; // Start year.
        $customfield['param2'] = date("Y")+5; // End year.
        $this->create_personal_goal_type_customfield('datetime', $customfield);
    }

    /**
     * Stub function to call create_personal_goal_type_customfield() with the correct
     * variables to create a checkbox type custom field
     *
     * @param array $data - The basic data to create the customfield with
     * @return void
     */
    public function create_personal_goal_type_checkbox($data) {
        $this->create_personal_goal_type_customfield('checkbox', $data);
    }

    /**
     * Stub function to call create_personal_goal_type_customfield() with the correct
     * variables to create a generic menu type custom field
     *
     * @param array $data - The basic data to create the customfield with
     * @return void
     */
    public function create_personal_goal_type_generic_menu($data) {
        $customfield = $data;
        $customfield['param1'] = str_replace(',', "\n", $data['value']);
        $customfield['value'] = '';
        $this->create_personal_goal_type_customfield('menu', $customfield);
    }

    /**
     * Create a custom field for a personal goal type
     * Note: While this can be called directly, it's easier to go through the
     *       setup functions above create_personal_goal_type_<cftype>()
     *
     * @param string $fieldtype  The type of customfield
     * @param array $customfield The data for the customfield, anything not provided will use default data
     */
    private function create_personal_goal_type_customfield($fieldtype, $customfield) {
        global $CFG, $DB;

        if (!$typeid = $DB->get_field('goal_user_type', 'id', array('idnumber' => $customfield['typeidnumber']))) {
            throw new coding_exception('Unknown personal_goal type idnumber '.$customfield['typeidnumber'].' in personal_goal definition');
        }

        $data = new \stdClass();
        $data->id = 0;
        $data->shortname = $fieldtype . $typeid;
        $data->typeid = $typeid;
        $data->datatype = $fieldtype;
        $data->description_editor = array('text' => '', 'format' => '1', 'itemid' => time());
        $data->hidden   = 0;
        $data->locked   = 0;
        $data->required = 0;
        $data->forceunique = 0;
        $data->defaultdata = $customfield['value'];
        if (isset($customfield['param1'])) {
            $data->param1 = $customfield['param1'];
        }
        if (isset($customfield['param2'])) {
            $data->param2 = $customfield['param2'];
        }
        if (isset($customfield['param3'])) {
            $data->param3 = $customfield['param3'];
        }
        if (isset($customfield['param4'])) {
            $data->param4 = $customfield['param4'];
        }
        if (isset($customfield['param5'])) {
            $data->param5 = $customfield['param5'];
        }
        $data->fullname  = 'Personal Goal ' . $fieldtype;

        require_once($CFG->dirroot . '/totara/customfield/field/' . $fieldtype . '/define.class.php');
        $customfieldclass = 'customfield_define_' . $fieldtype;
        $field = new $customfieldclass();
        $field->define_save($data, 'goal_user');
    }

    /**
     * Update a users scale value for an existing company goal assignment, and create
     * an associated history record for the change.
     *
     * @param int $userid
     * @param int $goalid
     * @param int $valueid - The id of the goal scale value record
     * @return boolean
     */
    public function update_company_goal_user_scale_value($userid, $goalid, $valueid): bool {
        $this->update_goal_user_scale_value('company', $userid, $goalid, $valueid);
        return true;
    }

    /**
     * Update a users scale value for an existing personal goal assignment, and create
     * an associated history record for the change.
     *
     * @param int $user_id
     * @param int $goal_id
     * @param int $scale_value_id - The id of the goal scale value record
     */
    public function update_personal_goal_user_scale_value(int $user_id, int $goal_id, int $scale_value_id): void {
        $this->update_goal_user_scale_value('personal', $user_id, $goal_id, $scale_value_id);
    }

    /**
     * Update a users scale value for an existing personal or company goal assignment, and create
     * an associated history record for the change.
     *
     * @param string $type
     * @param int $user_id
     * @param int $goal_id
     * @param int $scale_value_id - The id of the goal scale value record
     * @param int|null $timemodified time modified in timestamp format
     * @throws coding_exception
     */
    public function update_goal_user_scale_value(string $type, int $user_id, int $goal_id, int $scale_value_id, ?int $timemodified = null): int {
        global $DB, $USER;

        switch ($type) {
            case 'personal':
                $table = 'goal_personal';
                $conditions = ['userid' => $user_id, 'id' => $goal_id];
                $scope = goal::SCOPE_PERSONAL;
                break;
            case 'company':
                $table = 'goal_record';
                $conditions = ['userid' => $user_id, 'goalid' => $goal_id];
                $scope = goal::SCOPE_COMPANY;
                break;
            default:
                throw new coding_exception('Invalid goal type: ' . $type);
        }
        $record = $DB->get_record($table, $conditions);
        if (!$record) {
            throw new coding_exception("Could not find record in {$table} for user {$user_id} and goal {$goal_id}");
        }

        // Update the goal record with the new value_id.
        $record->scalevalueid = $scale_value_id;
        $DB->update_record($table, $record);

        // Create a history record for the change.
        $history = new \stdClass();
        $history->scope = $scope;
        $history->itemid = $record->id;
        $history->scalevalueid = $scale_value_id;
        $history->timemodified = $timemodified ?? time();
        $history->usermodified = $USER->id;
        return $DB->insert_record('goal_item_history', $history);
    }

    /**
     * Redirect behat generator with appropriate prefix.
     */
    public function create_pos($data) {
        return $this->create_hierarchy($data['frameworkid'], 'position', $data);
    }

    public function create_org($data) {
        global $DB;
        if (!empty($data['org_parent'])) {
            $data['parentid'] = $DB->get_field('org', 'id', ['idnumber' => $data['org_parent']], MUST_EXIST);
        }
        return $this->create_hierarchy($data['frameworkid'], 'organisation', $data);
    }

    public function create_comp($data) {
        return $this->create_hierarchy($data['frameworkid'], 'competency', $data);
    }

    public function create_goal($data, bool $trigger_event = true) {
        // Originally targetdate could be null, but since TL-35631 the column is
        // non nullable. Must catch any nulls passing through here.
        $data['targetdate'] = $data['targetdate'] ?? 0;

        return $this->create_hierarchy($data['frameworkid'], 'goal', $data, $trigger_event);
    }

    /**
     * Create hierarchy type.
     */
    public function create_pos_type($data = array()) {
        return $this->create_hierarchy_type('position', $data);
    }

    public function create_org_type($data = array()) {
        return $this->create_hierarchy_type('organisation', $data);
    }

    public function create_comp_type($data = array()) {
        return $this->create_hierarchy_type('competency', $data);
    }

    public function create_goal_type($data = array()) {
        return $this->create_hierarchy_type('goal', $data);
    }

    public function create_hierarchy_type($prefix, $data = array()) {
        global $USER, $DB;

        $shortprefix = $this->hierarchy_type_prefix[$prefix];

        $type = (object) $data;

        if (!isset($type->fullname)) {
            $type->fullname = 'Hierarchy ' . ucfirst($prefix) . ' Type ' . ($this->type_count[$prefix] + 1);
        }

        if (!isset($type->idnumber)) {
            $type->idnumber = totara_generator_util::create_short_name($type->fullname);
        }

        $type->description  = '';
        $type->timemodified = time();
        $type->usermodified = $USER->id;
        $type->timecreated  = time();

        $id = $DB->insert_record($shortprefix.'_type', $type);
        if (!$typeid = $DB->get_field($shortprefix.'_type', 'id', array('idnumber' => $type->idnumber))) {
            throw new coding_exception('Unknown hierarchy type idnumber '.$type->idnumber.' in hierarchy definition');
        }

        $this->type_count[$prefix]++;
        return $id;
    }

    public function create_hierarchy_type_menu($data) {
        $customfield = $data;
        $customfield['field']  = 'menu';
        $customfield['param1'] = "1234"."\n"."2345"."\n"."3456"."\n"."4567";
        $this->create_hierarchy_type_customfield($customfield);
    }

    public function create_hierarchy_type_text($data) {
        $customfield = $data;
        $customfield['field']  = 'text';
        $customfield['param1'] = 30;
        $customfield['param2'] = 2048;
        $this->create_hierarchy_type_customfield($customfield);
    }

    public function create_hierarchy_type_url($data) {
        $customfield = $data;
        $customfield['field']  = 'url';
        $this->create_hierarchy_type_customfield($customfield);
    }

    public function create_hierarchy_type_datetime($data) {
        $customfield = $data;
        $customfield['field']  = 'datetime';
        $customfield['param1'] = date("Y")-1; // Start year.
        $customfield['param2'] = date("Y")+5; // End year.
        //$customfield['param3'] = 1; // Include time. 0 for exclude.
        $this->create_hierarchy_type_customfield($customfield);
    }

    public function create_hierarchy_type_checkbox($data) {
        $customfield = $data;
        $customfield['field']  = 'checkbox';
        $this->create_hierarchy_type_customfield($customfield);
    }

    public function create_hierarchy_type_location($data, $locationdata = []) {
        $customfield = $data;
        $customfield['field']  = 'location';

        if (empty($customfield['param2']) && empty($locationdata)) {
            $locationdata = [
                "address" => "",
                "size" => "medium", // small, medium, large
                "view" => "map", // map, satellite, hybrid
                "display" => "address", // address, map, both
                "zoom" => 12,
                "location" => [
                    "latitude" => -36.866669999999999,
                    "longitude" => 174.76666
                ]
            ];
        }

        //$customfield['param2'] = json_encode($locationdata);
        $customfield['param2'] = $locationdata;
        $this->create_hierarchy_type_customfield($customfield);
    }

    public function create_hierarchy_type_textarea($data, $cols = null, $rows = null) {
        $customfield = $data;
        $customfield['field']  = 'textarea';

        if (empty($customfield['param1']) && empty($cols)) {
            $customfield['param1'] = 30;
        }
        if (empty($customfield['param2']) && empty($rows)) {
            $customfield['param2'] = 10;
        }

        $customfield['value'] = [
            'text' => $customfield['value'],
            'format' => '1'
        ];

        $this->create_hierarchy_type_customfield($customfield);
    }

    public function create_hierarchy_type_multiselect($data, $multiselectdata = []) {
        $customfield = $data;
        $customfield['field']  = 'multiselect';

        if (empty($customfield['param1']) && empty($multiselectdata)) {
            $multiselectdata = [[
                    "option" => "a",
                    "icon" => "",
                    "default" => "0",
                    "delete" => 0
                ],[
                    "option" => "b",
                    "icon" => "",
                    "default" => "0",
                    "delete" => 0
                ], [
                    "option" => "c",
                    "icon" => "",
                    "default" => "0",
                    "delete" => 0
                ], [
                    "option" => "d",
                    "icon" => "",
                    "default" => "0",
                    "delete" => 0
                ]
            ];
        }

        $customfield['param1'] = $multiselectdata;
        $this->create_hierarchy_type_customfield($customfield);
    }

    /**
     * Create a hierarchy type integer custom field.
     *
     * @param array $data The basic data to create the customfield with. Specify 'min', 'max' and 'step' to set the limits.
     * @return void
     */
    public function create_hierarchy_type_integer($data) {
        $customfield = $data;
        $customfield['field']  = 'integer';
        if (isset($data['min'])) {
            $customfield['param1'] = $data['min'];
        }
        if (isset($data['max'])) {
            $customfield['param2'] = $data['max'];
        }
        // Step must be set or else divide by zero errors occur.
        $customfield['param3'] = $data['step'] ?? 1;
        $this->create_hierarchy_type_customfield($customfield);
    }

    /**
     * A wrapped call to set_hierarchy_type_multiselect_data
     *
     * @param int    $itemid  - The id of the position
     * @param int    $fieldid - The id of the custom field
     * @param array  $data    - An array of the items to select
     * @return int
     */
    function set_pos_type_multiselect_data($posid, $fieldid, $data) {
        return $this->set_hierarchy_type_multiselect_data('position', $posid, $fieldid, $data);
    }

    /**
     * A wrapped call to set_hierarchy_type_multiselect_data
     *
     * @param int    $itemid  - The id of the organisation
     * @param int    $fieldid - The id of the custom field
     * @param array  $data    - An array of the items to select
     * @return int
     */
    function set_org_type_multiselect_data($orgid, $fieldid, $data) {
        return $this->set_hierarchy_type_multiselect_data('organisation', $orgid, $fieldid, $data);
    }

    /**
     * Create the data records for a hierarchy multiselect custom field.
     *
     * Note: For the best accuracy the data should be formatted with [hash => object]
     *       Where the object is an array holding all the selections data and the hash
     *       is an md5(object['option']). See hierarchy_type_customfield_mult
     *
     * @param string $prefix  - The hierarchy prefix position/organisation
     * @param int    $itemid  - The id of the hierarchy item
     * @param int    $fieldid - The id of the custom field
     * @param array  $data    - An array of the items to select
     * @return int
     */
    public function set_hierarchy_type_multiselect_data(string $prefix, int $itemid, int $fieldid, array $data) : int {
        global $DB;

        $shortprefix = $this->hierarchy_type_prefix[$prefix];
        $datatable = "{$shortprefix}_type_info_data";
        $paramtable = "{$shortprefix}_type_info_data_param";
        $fieldname = "{$prefix}id";

        $record = new \stdClass();
        $record->fieldid = $fieldid;
        $record->{$fieldname} = $itemid;
        $record->data = json_encode($data);
        $dataid = $DB->insert_record($datatable, $record);

        foreach ($data as $hash => $selection) {
            $msp_record = new \stdClass();
            $msp_record->dataid = $dataid;
            $msp_record->value = $hash;
            $DB->insert_record($paramtable, $msp_record);
        }

        return $dataid;
    }

    public function create_hierarchy_type_generic_menu($data) {
        $customfield = $data;
        $customfield['field']  = 'menu';
        $customfield['param1'] = str_replace(',', "\n", $data['value']);
        $customfield['value'] = '';
        $this->create_hierarchy_type_customfield($customfield);
    }

    private function create_hierarchy_type_customfield($customfield) {
        global $CFG, $DB;

        $datatype = $customfield['field'];
        $shortprefix = $this->hierarchy_type_prefix[$customfield['hierarchy']];
        $tableprefix = $shortprefix.'_type';
        if (!$typeid = $DB->get_field($tableprefix, 'id', array('idnumber' => $customfield['typeidnumber']))) {
            throw new coding_exception('Unknown hierarchy type idnumber '.$customfield['typeidnumber'].' in hierarchy definition');
        }

        $data = new \stdClass();
        $data->id = 0;
        $data->shortname = $datatype . $typeid;
        $data->fullname  = ucfirst($customfield['hierarchy']).' type '.$datatype;
        $data->typeid = $typeid;
        $data->datatype = $datatype;
        $data->description_editor = array('text' => '', 'format' => '1', 'itemid' => time());
        if ($datatype == 'textarea') {
            $data->defaultdata_editor = array('text' => '', 'format' => 0);
        }
        $data->hidden   = 0;
        $data->locked   = 0;
        $data->required = $customfield['required'] ?? 0;
        $data->forceunique = 0;
        $data->defaultdata = $customfield['value'];
        if (isset($customfield['param1'])) {
            $data->param1 = $customfield['param1'];
        }
        if (isset($customfield['param2'])) {
            $data->param2 = $customfield['param2'];
        }
        if (isset($customfield['param3'])) {
            $data->param3 = $customfield['param3'];
        }
        if (isset($customfield['param4'])) {
            $data->param4 = $customfield['param4'];
        }
        if (isset($customfield['param5'])) {
            $data->param5 = $customfield['param5'];
        }
        if ($datatype == 'multiselect' && isset($customfield['param1'])) {
            $data->multiselectitem = $customfield['param1'];
        }

        if (isset($customfield['shortname'])) {
            $data->shortname = $customfield['shortname'];
        }
        if (isset($customfield['fullname'])) {
            $data->fullname = $customfield['fullname'];
        }

        require_once($CFG->dirroot.'/totara/customfield/field/'.$datatype.'/define.class.php');
        $customfieldclass = 'customfield_define_'.$datatype;
        /** @var \customfield_define_base $field */
        $field = new $customfieldclass();
        $field->define_save($data, $tableprefix);
    }

    /**
     * Assign the requested hierarchy type to hierarchy
     *
     * @param array $data of prefix, hierarchy type custom field, hierarchy type id number, hierarchy id number, the value of custom field
     * @throws coding_exception
     */
    public function create_hierarchy_type_assign($data) {
        global $DB;

        // Pre-process any fields that require transforming.
        $shortprefix = hierarchy::get_short_prefix($data['hierarchy']);
        if (!$typeid = $DB->get_field($shortprefix.'_type', 'id', array('idnumber' => $data['typeidnumber']))) {
            throw new coding_exception('Unknown hierarchy type idnumber '.$data['typeidnumber'].' in hierarchy definition');
        }
        $DB->set_field($shortprefix, 'typeid', $typeid, array('idnumber' => $data['idnumber']));
        if (!$hierarchyid = $DB->get_field($shortprefix, 'id', array('idnumber' => $data['idnumber']))) {
            throw new coding_exception('Unknown hierarchy idnumber '.$data['idnumber'].' in hierarchy definition');
        }
        $field = $data['field'];
        $input = "customfield_{$field}{$typeid}";
        if ($field == 'textarea') {
            $input .= '_editor';
        }
        $item = new \stdClass();
        $item->id = $hierarchyid;
        $item->typeid = $typeid;

        if ($field == 'location') {
            if (isset($data['value']['address'])) {
                $item->{$input.'address'} = $data['value']['address'];
            }
            if (isset($data['value']['size'])) {
                $item->{$input.'size'} = $data['value']['size'];
            }
            if (isset($data['value']['view'])) {
                $item->{$input.'view'} = $data['value']['view'];
            }
            if (isset($data['value']['display'])) {
                $item->{$input.'display'} = $data['value']['display'];
            }
            if (isset($data['value']['location']['latitude'])) {
                $item->{$input . 'latitude'} = $data['value']['latitude'];
            }
            if (isset($data['value']['location']['longitude'])) {
                $item->{$input . 'longitude'} = $data['value']['longitude'];
            }
        } else {
            $item->{$input} = $data['value'];
        }
        customfield_save_data($item, $data['hierarchy'], $shortprefix.'_type');
    }

    /**
     * Create a hierarchy based on the shortprefix and assign it to a framework.
     *
     * @param $frameworkid
     * @param $prefix
     * @param null $record
     * @param bool $trigger_event Trigger event flag - set to off to reduce overhead.
     * @return stdClass hierarchy item
     *
     * @todo Define an array of default values then use
     *       array_merge($default_values,$record) to
     *       merge in the optional record data and reduce
     *       / remove the need for multiple statements
     *       beginning with: if (!isset($record['...
     */
    public function create_hierarchy($frameworkid, $prefix, $record = null, bool $trigger_event = true) {
        global $DB, $USER, $CFG;
        require_once($CFG->dirroot . '/totara/hierarchy/lib.php');

        if (is_string($record)) {
            throw new coding_exception('$record parameter must be array or object');
        }

        $record = (array) $record;
        // Increment the count for the given hierarchy.
        $i = ++$this->hierarchycount[$prefix];

        if (!isset($record['fullname'])) {
            $defaultnameconst = 'self::DEFAULT_NAME_HIERARCHY_' . strtoupper($prefix);
            $record['fullname'] = trim(constant($defaultnameconst)) . ' ' .$i;
        }

        if (!isset($record['idnumber'])) {
            $record['idnumber'] = totara_generator_util::create_short_name($record['fullname']);
        }

        if (!isset($record['description'])) {
            $record['description'] = '<p>' . $record['fullname'] . ' description</p>';
        }

        if (!isset($record['visible'])) {
            $record['visible'] = 1;
        }

        if (!isset($record['hidecustomfields'])) {
            $record['hidecustomfields'] = 0;
        }

        if (!isset($record['parentid'])) {
            $record['parentid'] = 0;
        }

        if (!isset($record['aggregationmethod'])) {
            // Get a default value for the agreggation method.
            // This variable is used to build the select menu
            // in the hierarchy form.
            $record['aggregationmethod'] = competency::AGGREGATION_METHOD_ALL;
        }

        if (!isset($record['proficiencyexpected'])) {
            // The default value for proficiencyexpected
            // is hard coded in the hierarchy form.
            $record['proficiencyexpected'] = 1;
        }

        if (!isset($record['timecreated'])) {
            $record['timecreated'] = time();
        }

        $record['frameworkid'] = $frameworkid;
        $record['timemodified'] = $record['timecreated'];
        $record['usermodified'] = $USER->id;

        $record = (object) $record;
        $hierarchy = hierarchy::load_hierarchy($prefix);
        $itemnew = $hierarchy->process_additional_item_form_fields($record);
        $item = $hierarchy->add_hierarchy_item($itemnew, $itemnew->parentid, $itemnew->frameworkid, false, $trigger_event, false);

        return $item;
    }

    /**
     * Create some hierarchies.
     *
     * @param int $frameworkid The framework to assign the hierachies to.
     * @param string $prefix The type of hierarchy to create.
     * @param int $quantity The number of hierarchies to create.
     * @param string $name The base name of the hierarchy.
     * @param int $randomise_percent Randomly determine (by percentage) if the hierarchy is created.
     * @param array $hierarchy_extra_data
     * @param bool $trigger_event Trigger event flag, turn off to reduce overhead on a large amount of data
     * @return array of hierarchies
     * @throws coding_exception
     */
    public function create_hierarchies($frameworkid, $prefix, $quantity, $name = '', $randomise_percent = 0, $hierarchy_extra_data = [], bool $trigger_event = true) {
        global $CFG;
        require_once($CFG->dirroot . '/totara/hierarchy/lib.php');

        // Create the objective name we want to use with by ge3tting
        // the number off any previous matching records we created.
        if (!$name) {
            $name = 'self::DEFAULT_NAME_HIERARCHY_' . strtoupper($prefix);
        }
        $shortprefix = hierarchy::get_short_prefix($prefix);
        $number = totara_generator_util::get_next_record_number($shortprefix, 'fullname', $name);

        $hierarchy_data = array ();
        $hierarchy_ids = array ();
        // Create the quantity of hierarchies we need.
        for ($i = 1; $i <= $quantity; $i++) {
            // Create a hierarchy, or apply randomisation and create if required.
            if ($randomise_percent == 0 || ($randomise_percent && get_random_act($randomise_percent))) {
                $hierarchy_data['fullname'] = $name . ' ' . $number++;
                $create_data = array_merge ($hierarchy_data, $hierarchy_extra_data);
                $hierarchy = $this->create_hierarchy($frameworkid, $prefix, $create_data, $trigger_event);
                $hierarchy_ids[$i] = $hierarchy->id;
            }
        }

        return $hierarchy_ids;
    }

    /**
     * Assign linked course to a competency.
     *
     * @param stdClass $competency Competency to add linked course to
     * @param stdClass $course Course to add
     *
     * @return int
     */
    public function assign_linked_course_to_competency($competency, $course) {
        global $CFG, $DB;

        $evidence = \competency_evidence_type::factory(array('itemtype' => 'coursecompletion'));

        $evidence->iteminstance = $course->id;
        $newevidenceid = $evidence->add($competency);

        return $newevidenceid;
    }

    /**
     * Remove linked course from a competency.
     *
     * @param stdClass $competency Competency to remove linked course from
     * @param stdClass $course Course to remove
     *
     * @return true
     */
    public function remove_linked_course_from_competency($competency, $evidenceid) {
        /** @var \competency_evidence_type $evidence */
        $evidence = \competency_evidence_type::factory($evidenceid);
        $evidence->delete($competency);

        return true;
    }


    /**
     * Assigns random courses to competencies.
     *
     * @param $size int number of items to process.
     */
    public function assign_competency($size) {
        global $CFG, $DB;
        require_once($CFG->dirroot . '/totara/hierarchy/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/competency/lib.php');

        // Do not assign the site course!
        $site = get_site();
        // Track the ones we have already assigned.
        $assignedhierarchies = array();
        // Get $size competencies.
        for ($x=0; $x < $size; $x++) {
            // Find one we have not already used.
            $hierarchyid = totara_generator_util::get_random_record_id($this->hierarchy_type_prefix['competency'], false, $assignedhierarchies);
            $assignedhierarchies[] = $hierarchyid;
            // Load competency
            if ($competency = $DB->get_record('comp', array('id' => $hierarchyid))) {
                // Assign random number of courses up to $size.
                $coursesassigned = 0;
                $coursestoassign = mt_rand(0, $size);
                $tries = 0;
                while ($coursesassigned < $coursestoassign) {
                    // Set up the completion evidence type.
                    $evidence = \competency_evidence_type::factory(array('itemtype' => 'coursecompletion'));
                    $evidence->iteminstance = totara_generator_util::get_random_record_id('course');
                    if ($evidence->iteminstance != $site->id && !$DB->record_exists('comp_criteria', array('competencyid' => $hierarchyid, 'itemtype' => 'coursecompletion', 'iteminstance' => $evidence->iteminstance))) {
                        // Randomise mandatory or optional.
                        $evidence->linktype = mt_rand(0,1);
                        // Assign courses to competency.
                        $newevidenceid = $evidence->add($competency);
                        $coursesassigned++;
                        $tries = 0;
                    } else {
                        $tries++;
                        if ($tries > 128) {
                            // This may cause intermittent failure due to get_random_record_id() not choosing an
                            //   unassigned course 129 times in a row.
                            throw new \moodle_exception("Unable to find an unassigned course for the competency, giving up.");
                        }
                    }
                }
            }
        }
        echo "\n" . get_string('progress_assigncoursecompetencies', 'totara_generator');
    }

    /**
     * Assigns random competencies and goals to organisations.
     *
     * @param $size int number of items to process.
     */
    public function assign_organisation($size) {
        $this->assign_competency_to_hierarchy('organisation', $size);
    }

    /**
     * Assigns random competencies and goals to positions
     *
     * @param $size int number of items to process.
     */
    public function assign_position($size) {
        $this->assign_competency_to_hierarchy('position', $size);
    }

    /**
     * Assigns random user groups to goals.
     *
     * @param $size int number of items to process.
     */
    public function assign_goal($size) {
        global $CFG, $DB;
        require_once($CFG->dirroot . '/totara/hierarchy/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/goal/assign/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/goal/lib.php');

        // Track the ones we have already assigned.
        $assignedhierarchies = array();
        // Get all goals to assign at least one group.
        for ($x=0; $x < $size; $x++) {
            // Find one we have not already used.
            $hierarchyid = totara_generator_util::get_random_record_id($this->hierarchy_type_prefix['goal'], false, $assignedhierarchies);
            $assignedhierarchies[] = $hierarchyid;
            // Get the base goal item.
            $item = $DB->get_record('goal', array('id' => $hierarchyid));
            $baseclassname = "totara_assign_goal";
            $baseclass = new \totara_assign_goal('goal', $item);
            // Assign random pos, org or cohort groups to this goal.
            $grouptypes = array('pos', 'org', 'cohort');
            $groupstoassign = mt_rand(1,3);
            for ($i=0; $i < $groupstoassign; $i++) {
                $grouptype = $grouptypes[mt_rand(0,2)];
                $grouptypeobj = $baseclass->load_grouptype($grouptype);
                // Get a random record from the groups.
                $groupid = totara_generator_util::get_random_record_id($grouptype);
                $grouptypeobj->validate_item_selector($groupid);
                $urlparams = array('module' => 'group',
                        'grouptype' => $grouptype,
                        'itemid' => $hierarchyid,
                        'add' => 1,
                        'listofvalues' => array($groupid),
                        'includechildren' => 0
                );
                $grouptypeobj->handle_item_selector($urlparams);
            }
        }
        echo "\n" . get_string('progress_assigngoalusergroups', 'totara_generator');
    }

    /**
     * Assigns a random number of competencies from 1 to $size to each item in the hierarchy type
     *
     * @param $hierarchytype string usually 'position' or 'organisation'
     * @param $size int number of items to process.
     */
    private function assign_competency_to_hierarchy($hierarchytype, $size) {
        global $CFG, $DB, $USER;
        require_once($CFG->dirroot . '/totara/hierarchy/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/competency/lib.php');
        require_once($CFG->dirroot . '/totara/hierarchy/prefix/' . $hierarchytype . '/lib.php');

        $time = time();
        // Track the ones we have already assigned.
        $assignedhierarchies = array();
        // Assign something to every hierarchy item.
        for ($x=0; $x < $size; $x++) {
            // Find one we have not already used.
            $hierarchyid = totara_generator_util::get_random_record_id($this->hierarchy_type_prefix[$hierarchytype], false, $assignedhierarchies);
            $assignedhierarchies[] = $hierarchyid;
            // Setup hierarchy objects
            $competencies = new competency();
            $hierarchies = new $hierarchytype();
            // Load position
            if (!$hierarchy = $hierarchies->get_item($hierarchyid)) {
                print_error("{$hierarchytype}notfound", 'totara_hierarchy');
            }
            // Currently assigned competencies
            if (!$currentlyassigned = $hierarchies->get_assigned_competencies($hierarchyid)) {
                $currentlyassigned = array();
            }
            $addcompetencies = 0;
            $add = array();
            $competencytoassign = mt_rand(1, $size);
            while ($addcompetencies < $competencytoassign) {
                $newcomp = totara_generator_util::get_random_record_id ($this->hierarchy_type_prefix['competency'], false, $currentlyassigned);
                $add[] = $newcomp;
                // Add it to currently assigned too - on small sites it may try to add the same competency twice.
                $currentlyassigned[] = $newcomp;
                $addcompetencies++;
            }
            foreach ($add as $addition) {
                // Add relationship
                $related = $competencies->get_item($addition);
                $relationship = new stdClass();
                $field = "{$hierarchytype}id";
                $relationship->$field = $hierarchy->id;
                $relationship->competencyid = $related->id;
                $relationship->timecreated = $time;
                $relationship->usermodified = $USER->id;
                $relationship->linktype = mt_rand(0,1);
                $relationship->id = $DB->insert_record($this->hierarchy_type_prefix[$hierarchytype] . '_competencies', $relationship);
            }
        }
        echo "\n" . get_string('progress_assigncompetenciestohierarchy', 'totara_generator', get_string($hierarchytype, 'totara_hierarchy'));
    }

    /**
     * Wrapper for behat for creating and aging a goal.
     * Works for both personal and company goals.
     *
     * @param array $data
     * @return void
     */
    public function create_goal_with_backdate(array $data): void {
        /** @var user $user */
        $user = user::repository()
            ->where('username', $data['username'])
            ->one(true);

        $backdate_time = isset($data['backdate']) ? strtotime($data['backdate']) : null;
        $data['targetdate'] = isset($data['targetdate']) ? strtotime($data['targetdate']) : null;

        if ($data['scope'] === 'company') {
            $this->create_company_goal_and_backdate($user->to_record(), $backdate_time, $data);
        } else if ($data['scope'] === 'personal') {
            $this->create_personal_goal_and_backdate($user->to_record(), $backdate_time, $data);
        } else {
            throw new coding_exception("Unknown goal scope: " . $data['scope']);
        }
    }

    /**
     * Create a personal goal for a user and age it.
     *
     * @param stdClass $user
     * @param int|null $backdate_time
     * @param array $goal_data
     * @return personal_goal_entity
     */
    public function create_personal_goal_and_backdate(
        stdClass $user,
        ?int $backdate_time,
        array $goal_data = []
    ): personal_goal_entity {

        $backdate_time = $backdate_time ?? time();

        $goaldata = [
            'name' => $goal_data['name'] ?? "pg-" . uniqid('', true),
            'assigntype' => personal_goal_assignment_type::self()->get_value(),
            'scaleid' => $goal_data['scaleid'] ?? 1,
            'scalevalueid' => $goal_data['scalevalueid'] ?? 1,
            'timecreated' => $backdate_time,
            'timemodified' => $backdate_time,
            'targetdate' => $goal_data['targetdate'] ?? 0,
            'description' => $goal_data['description'] ?? null,
        ];
        $goal_id = $this->create_personal_goal($user->id, $goaldata)->id;

        // Make sure we only have that one record in the history table.
        $goal_item_history_id = goal_item_history::repository()
            ->where('itemid', $goal_id)
            ->where('scope', goal::SCOPE_PERSONAL)
            ->one(true)
            ->id;

        goal_item_history::repository()
            ->where('id', $goal_item_history_id)
            ->update(['timemodified' => $backdate_time]);

        return new personal_goal_entity($goal_id);
    }

    /**
     * Create a company goal for a user and age it.
     *
     * @param stdClass $user
     * @param int|null $backdate_time
     * @param array $goal_data
     * @return company_goal_entity
     */
    public function create_company_goal_and_backdate(
        stdClass $user,
        ?int $backdate_time,
        array $goal_data = []
    ): company_goal_entity {

        $backdate_time = $backdate_time ?? time();

        $fw_id = $this->create_goal_frame(['name' => 'fw'])->id;
        $goal_id = $this->create_goal(
            [
                'fullname' => $goal_data['name'] ?? "cg-" . uniqid('', true),
                'frameworkid' => $fw_id,
                'timecreated' => $backdate_time,
                'targetdate' => $goal_data['targetdate'] ?? 0,
                'description' => $goal_data['description'] ?? null,
            ]
        )->id;

        // The following line includes backdating goal_user_assignment.timemodified
        $this->goal_assign_individuals($goal_id, [$user->id], $backdate_time);

        $item_id = $this->get_goal_item_id($goal_id, $user->id, goal::SCOPE_COMPANY);

        // Make sure we only have that one record in the history table.
        $goal_item_history_id = goal_item_history::repository()
            ->where('itemid', $item_id)
            ->where('scope', goal::SCOPE_COMPANY)
            ->one(true)
            ->id;

        goal_item_history::repository()
            ->where('id', $goal_item_history_id)
            ->update(['timemodified' => $backdate_time]);

        return new company_goal_entity($goal_id);
    }

    /**
     * Get a goal item id for a goal/user combination.
     * Works for both personal and company goals.
     *
     * @param $goal_id
     * @param $user_id
     * @param int $scope
     * @return int
     */
    public static function get_goal_item_id($goal_id, $user_id, int $scope): int {
        if ($scope === goal::SCOPE_COMPANY) {
            return company_goal_assignment::repository()
                ->where('userid', $user_id)
                ->where('goalid', $goal_id)
                ->one(true)
                ->id;
        }

        if ($scope === goal::SCOPE_PERSONAL) {
            return $goal_id;
        }

        throw new coding_exception('Invalid goal scope: ' . $scope);
    }

    /**
     * Update a goal to a given scale value for a user and backdate that update (goal_item_history record) to a given date.
     * Works for both personal and company goals.
     *
     * @param int $goal_id
     * @param int $user_id
     * @param int $scope
     * @param int $backdate_time
     * @param string $scale_value_name
     * @return void
     */
    public function progress_goal_and_backdate(
        int $goal_id,
        int $user_id,
        int $scope,
        int $backdate_time,
        string $scale_value_name = 'Goal in progress'
    ): void {
        $item_id = $this->progress_goal($goal_id, $user_id, $scope, $scale_value_name);

        // Backdate the latest history record
        $goal_item_history_id = goal_item_history::repository()
            ->where('itemid', $item_id)
            ->where('scope', $scope)
            ->order_by('id')
            ->get()
            ->last()
            ->id;

        goal_item_history::repository()
            ->where('id', $goal_item_history_id)
            ->update(['timemodified' => $backdate_time]);
    }

    /**
     * Update a goal to a given scale value for a user.
     * Works for both personal and company goals.
     *
     * @param int $goal_id
     * @param int $user_id
     * @param int $scope
     * @param string $scale_value_name
     * @return int
     */
    private function progress_goal(int $goal_id, int $user_id, int $scope, string $scale_value_name = 'Goal in progress'): int {
        $scale_value_id = builder::table('goal_scale_values')
            ->where('name', $scale_value_name)
            ->one(true)
            ->id;

        $to_db = new stdClass();
        $item_id = $this->get_goal_item_id($goal_id, $user_id, $scope);
        $to_db->id = $item_id;
        $to_db->scalevalueid = $scale_value_id;
        goal::update_goal_item($to_db, $scope);

        return $item_id;
    }
}
