<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2021 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Qingyang Liu <qingyang.liu@totaralearning.com>
 * @package totara_oauth2
 */

namespace totara_oauth2\model;

use coding_exception;
use context_system;
use core\collection;
use core\entity\tenant;
use core\orm\entity\encrypted_model;
use core\orm\query\builder;
use dml_exception;
use dml_transaction_exception;
use totara_oauth2\config;
use totara_oauth2\entity\access_token;
use totara_oauth2\entity\client_provider as entity;
use totara_oauth2\event\rotated_secret_event;
use totara_oauth2\grant_type;

/**
 *
 * @property-read int $id
 * @property-read string $client_id
 * @property-read string $client_secret
 * @property-read string $id_number
 * @property-read string $name
 * @property-read string|null $description
 * @property-read int|null $description_format
 * @property-read string|null $scope
 * @property-read string|null $grant_types
 * @property-read int $internal
 * @property-read int $time_created
 * @property-read bool $status
 * @property-read int|null tenant_id
 * @property-read tenant|null $tenant_entity
 * @property-read string|null $component
 * @property-read int|null $client_secret_updated_at
 * @property-read access_token[]|collection $access_tokens
 *
 * @property-read string $detail_scope
 */
class client_provider extends encrypted_model {
    /**
     * @var entity
     */
    protected $entity;

    /**
     * @var string[]
     */
    protected $entity_attribute_whitelist = [
        'id',
        'client_id',
        'client_secret',
        'id_number',
        'name',
        'description',
        'description_format',
        'scope',
        'grant_types',
        'internal',
        'time_created',
        'status',
        'tenant_id',
        'component',
        'client_secret_updated_at',
        'access_tokens',
    ];

    /**
     * @var string[]
     */
    protected $model_accessor_whitelist = [
        'tenant_entity',
        'detail_scope',
    ];

    protected $encrypted_attribute_list = [
        'client_secret',
    ];

    /**
     * @return string
     */
    protected static function get_entity_class(): string {
        return entity::class;
    }

    /**
     * @return string|null
     */
    public function get_detail_scope(): ?string {
        switch ($this->scope) {
            case config::XAPI_WRITE: {
                return get_string('xapi_write', 'totara_oauth2');
            }
            default: {
                return null;
            }
        }
    }

    /**
     * @param string $name
     * @param string $scope_type
     * @param int $format
     * @param string|null $description
     * @param int $internal
     * @param bool $status
     * @param int|null $tenant_id
     * @param string|null $component
     * @return static
     * @throws coding_exception
     * @throws dml_transaction_exception|dml_exception
     */
    public static function create(
        string $name,
        string $scope_type,
        int $format,
        string $description = null,
        int $internal = 0,
        bool $status = true,
        int $tenant_id = null,
        string $component = null
    ): self {
        $entity = new entity();
        $entity->client_secret_updated_at = time();
        $entity->client_id = self::generate_client_id();
        $entity->client_secret = random_string(24);
        $entity->name = $name;
        $entity->description_format = $format;
        $entity->description = $description ?? '';
        $entity->grant_types = grant_type::get_client_credentials();
        $entity->internal = $internal;
        $entity->scope = $scope_type;
        $entity->status = $status;
        $entity->component = $component;

        if (!empty($tenant_id)) {
            $entity->tenant_id = $tenant_id;
        }

        $db = builder::get_db();
        $transaction = $db->start_delegated_transaction();
        $entity->save();
        $model = static::load_by_entity($entity);
        $model->set_encrypted_attribute('client_secret', $model->client_secret);
        $db->commit_delegated_transaction($transaction);

        return $model;
    }

    /**
     * Generates a unique client ID
     * @return string The unique client ID
     * @throws dml_exception
     * @throws coding_exception
     */
    private static function generate_client_id(): string {
        $db = builder::get_db();

        $attempts = 0;
        do {
            if ($attempts >= config::MAX_GENERATION_ATTEMPTS) {
                throw new coding_exception("A unique client ID could not be auto generated");
            }

            $client_id = random_string(16);
            $attempts++;
        } while ($db->record_exists(entity::TABLE, ['client_id' => $client_id]));

        return $client_id;
    }

    /**
     * @return void
     */
    public function delete(): void {
        $this->entity->delete();
    }

    /**
     * Get tenant.
     *
     * @return tenant|null
     */
    public function get_tenant_entity(): ?tenant {
        return $this->entity->tenant;
    }

    /**
     * Rotates the client secret
     *
     * @return client_provider
     */
    public function rotate_secret(): client_provider {
        global $DB, $USER;

        $access_tokens = $this->entity->access_tokens();
        $access_tokens_count = $access_tokens->count();

        $DB->transaction(function () use ($access_tokens) {
            // Hard-delete any existing tokens associated with the client
            $access_tokens->delete();

            // Rotate the secret
            $this->set_encrypted_attribute('client_secret', random_string(24));
            $this->entity->client_secret_updated_at = time();
            $this->entity->update();
        });

        $event = rotated_secret_event::create([
            'objectid' => $this->entity->id,
            'userid' => $USER->id,
            'context' => context_system::instance(),
            'other' => [
                'access_tokens_count' => $access_tokens_count,
            ],
        ]);
        $event->trigger();

        return $this;
    }
}
