<?php
/**
 * This file is part of Totara Learn
 *
 * Copyright (C) 2023 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Nathaniel Walmsley <nathaniel.walmsley@totara.com>
 * @package totara_program
 */

namespace totara_program\totara_notification\resolver;

use context_program;
use core\orm\query\builder;
use core_user\totara_notification\placeholder\user;
use core_user\totara_notification\placeholder\users;
use lang_string;
use totara_core\extended_context;
use totara_core\identifier\component_area;
use totara_job\job_assignment;
use totara_notification\placeholder\placeholder_option;
use totara_notification\resolver\abstraction\additional_criteria_resolver;
use totara_notification\resolver\abstraction\audit_resolver;
use totara_notification\resolver\abstraction\permission_resolver;
use totara_notification\resolver\notifiable_event_resolver;
use totara_notification\supports_context_helper;
use totara_program\totara_notification\placeholder\assignment;
use totara_program\totara_notification\placeholder\extension_request as extension_request;
use totara_program\totara_notification\placeholder\program;
use totara_notification\recipient\manager;
use totara_notification\recipient\subject;

class extension_resolved extends notifiable_event_resolver implements permission_resolver, audit_resolver, additional_criteria_resolver {

    /**
     * @inheritDoc
     */
    public static function get_notification_title(): string {
        return get_string('notification_response_to_extension_request_title', 'totara_program');
    }

    /**
     * @inheritDoc
     */
    public static function get_notification_available_recipients(): array {
        return [
            subject::class,
            manager::class,
        ];
    }

    /**
     * @inheritDoc
     */
    public static function get_notification_default_delivery_channels(): array {
        return ['email', 'popup'];
    }

    /**
     * @inheritDoc
     */
    public static function get_notification_available_placeholder_options(): array {
        return [
            placeholder_option::create(
                'program',
                program::class,
                new lang_string('notification_program_placeholder_group', 'totara_program'),
                function (array $event_data): program {
                    return program::from_id($event_data['program_id']);
                }
            ),
            placeholder_option::create(
                'completion',
                assignment::class,
                new lang_string('notification_completion_placeholder_group', 'totara_program'),
                function (array $event_data): assignment {
                    return assignment::from_program_id_and_user_id($event_data['program_id'], $event_data['subject_user_id']);
                }
            ),
            placeholder_option::create(
                'recipient',
                user::class,
                new lang_string('notification_recipient_placeholder_group', 'totara_program'),
                function (array $event_data, int $target_user_id): user {
                    return user::from_id($target_user_id);
                }
            ),
            placeholder_option::create(
                'managers',
                users::class,
                new lang_string('notification_manager_placeholder_group', 'totara_program'),
                function (array $event_data): users {
                    return users::from_ids(job_assignment::get_all_manager_userids($event_data['subject_user_id']));
                }
            ),
            placeholder_option::create(
                'resolver',
                user::class,
                new lang_string('notification_resolver_placeholder_group', 'totara_program'),
                function (array $event_data): user {
                    return user::from_id($event_data['resolver_user_id']);
                }
            ),
            placeholder_option::create(
                'subject',
                user::class,
                new lang_string('notification_subject_placeholder_group', 'totara_program'),
                function (array $event_data): user {
                    return user::from_id($event_data['subject_user_id']);
                }
            ),
            placeholder_option::create(
                'extension_request',
                extension_request::class,
                new lang_string('notification_extension_request_placeholder_group', 'totara_program'),
                function (array $event_data): extension_request {
                    return extension_request::from_id($event_data['extension_request_id']);
                }
            )
        ];
    }

    /**
     * @inheritDoc
     */
    public function get_extended_context(): extended_context {
        return extended_context::make_with_context(
            context_program::instance($this->event_data['program_id']),
            'totara_program',
            'program',
            $this->event_data['program_id']
        );
    }

    /**
     * @inheritDoc
     */
    public static function uses_on_event_queue(): bool {
        return true;
    }

    /**
     * @inheritDoc
     */
    public static function supports_context(extended_context $extended_context): bool {
        // We need to do an extra check to make sure we don't mix up programs and certs.
        if ($extended_context->is_natural_context() && $extended_context->get_context_level() == CONTEXT_PROGRAM) {
            $prog = builder::table('prog')
                ->select('certifid')
                ->where('id', '=', $extended_context->get_context()->instanceid)
                ->one(true);
            return empty($prog->certifid);
        }

        return supports_context_helper::supports_context(
            $extended_context,
            CONTEXT_PROGRAM,
            'container_course',
            null,
            new component_area('totara_program', 'program')
        );
    }

    /**
     * @inheritDoc
     */
    public static function can_user_manage_notification_preferences(extended_context $context, int $user_id): bool {
        $natural_context = $context->get_context();
        return has_capability('totara/program:configuremessages', $natural_context, $user_id);
    }

    /**
     * @inheritDoc
     */
    public static function can_user_audit_notifications(extended_context $context, int $user_id): bool {
        $natural_context = $context->get_context();
        return has_capability('totara/program:auditmessages', $natural_context, $user_id);
    }

    /**
     * @inheritDoc
     */
    public function get_notification_log_display_string_key_and_params(): array {
        // The resolver title is translated at view time
        $params = ['resolver_title' => ''];

        $user = user::from_id($this->get_event_data()['subject_user_id']);
        $params['user'] = $user->do_get('full_name');

        $program = program::from_id($this->get_event_data()['program_id']);
        $params['program'] = $program->do_get('full_name');

        return [
            'key' => 'notification_log_extension_request_resolver',
            'component' => 'totara_program',
            'params' => $params,
        ];
    }

    /**
     * @inheritDoc
     */
    public static function get_additional_criteria_component(): string {
        return 'totara_program/components/notification/ExtensionRequestTriggers';
    }

    /**
     * @inheritDoc
     */
    public static function is_valid_additional_criteria(
        array $additional_criteria, extended_context $extended_context
    ): bool {
        if (
            !isset($additional_criteria['extensionRequestTriggers']) ||
            !is_array($additional_criteria['extensionRequestTriggers'])
        ) {
            return false;
        }
        $expected = ["request_granted", "request_denied"];
        // Check that all elements in the additional criteria array are in the expected array
        if (!empty(array_diff($additional_criteria['extensionRequestTriggers'], $expected))) {
            return false;
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public static function meets_additional_criteria(?array $additional_criteria, array $event_data): bool {
        $extension_request = extension_request::from_id($event_data['extension_request_id']);
        $status = $extension_request->get_status_int();
        // Ignore status 0 (created). We will never send this notification for an incomplete request
        if ($status === 0) {
            return false;
        }

        $status_map = [
            'request_granted' => 1,
            'request_denied' => 2
        ];

        $mapped_triggers = [];
        foreach ($additional_criteria['extensionRequestTriggers'] as $trigger) {
            $mapped_triggers[] = $status_map[$trigger];
        }

        if (!in_array($status, $mapped_triggers)) {
            return false;
        }

        return true;
    }
}