<?php
/*
 * This file is part of Totara LMS
 *
 * Copyright (C) 2016 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Brendan Cox <brendan.cox@totaralms.com>
 * @package totara_program
 */

defined('MOODLE_INTERNAL') || die();

use totara_program\content\program_content;
use totara_program\program;

global $CFG;

/**
 * Class totara_program_program_content_testcase
 *
 * Tests methods found within the prog_content class.
 *
 * To test, run this from the command line from the $CFG->dirroot.
 * vendor/bin/phpunit --verbose totara_program_program_content_testcase totara/program/tests/program_content_test.php
 */
class totara_program_program_content_test extends \core_phpunit\testcase {

    /** @var \core\testing\generator */
    private $generator;

    /** @var \totara_program\testing\generator*/
    private $program_generator;

    /** @var stdClass */
    private $course1, $course2, $course3;

    /** @var program */
    private $program1, $program2;

    protected function tearDown(): void {
        $this->generator = null;
        $this->program_generator = null;
        $this->course1 = $this->course2 = $this->course3 = null;
        $this->program1 = $this->program2 = null;

        parent::tearDown();
    }

    public function setUp(): void {
        parent::setUp();
        global $DB;

        $this->generator = $this->getDataGenerator();
        $this->program_generator = $this->generator->get_plugin_generator('totara_program');

        $this->course1 = $this->generator->create_course();
        $this->course2 = $this->generator->create_course();
        $this->course3 = $this->generator->create_course();
        $this->program1 = $this->program_generator->create_program();
        $this->program2 = $this->program_generator->create_program();

        // Reload courses. Otherwise when we compare the courses with the returned courses,
        // we get subtle differences in some values such as cacherev and sortorder.
        // Todo: Investigate whether we can improve the generator to fix this.
        $this->course1 = $DB->get_record('course', array('id' => $this->course1->id));
        $this->course2 = $DB->get_record('course', array('id' => $this->course2->id));
        $this->course3 = $DB->get_record('course', array('id' => $this->course3->id));
    }

    /**
     * Tests the get_courseset_by_id method.
     */
    public function test_get_courseset_by_id() {
        global $DB;

        $progcontent = new program_content($this->program1->id);
        $progcontent->add_set(program_content::CONTENTTYPE_MULTICOURSE);
        $progcontent->add_set(program_content::CONTENTTYPE_MULTICOURSE);
        $progcontent->add_set(program_content::CONTENTTYPE_COMPETENCY);

        /** @var course_set[] $coursesets */
        $coursesets = $progcontent->get_course_sets();

        // We'll add courses while we know what the sortorder is.
        $coursedata = new stdClass();
        $coursedata->{$coursesets[0]->get_set_prefix() . 'courseid'} = $this->course1->id;
        $progcontent->add_course(1, $coursedata);

        $coursedata->{$coursesets[1]->get_set_prefix() . 'courseid'} = $this->course2->id;
        $progcontent->add_course(2, $coursedata);

        /** @var \totara_hierarchy\testing\generator $hierarchygenerator */
        $hierarchygenerator = $this->generator->get_plugin_generator('totara_hierarchy');
        $competencyframework = $hierarchygenerator->create_comp_frame(array());
        $competencydata = array('frameworkid' => $competencyframework->id);
        $competency = $hierarchygenerator->create_comp($competencydata);
        // Completions for course 3 will be assigned to this competency.
        $course3evidenceid = $hierarchygenerator->assign_linked_course_to_competency($competency, $this->course3);

        // Add a competency to the competency courseset.
        $compdata = new stdClass();
        $compdata->{$coursesets[2]->get_set_prefix() . 'competencyid'} = $competency->id;
        $progcontent->add_competency(3, $compdata);

        // Now try to get coursesets by id before saving using an arbitrary id number. This should return false.
        $this->assertEquals(false, $progcontent->get_courseset_by_id(1));

        $progcontent->save_content();

        // Get the ids. At this stage we know what the sortorder should be and can test based on the below records.
        // Multi course set which contains course1.
        $courseset1 = $DB->get_record('prog_courseset', array('programid' => $this->program1->id, 'sortorder' => 1));
        // Multi course set which contains course2.
        $courseset2 = $DB->get_record('prog_courseset', array('programid' => $this->program1->id, 'sortorder' => 2));
        // Competency course set which contains competency 1 which links to course3.
        $courseset3 = $DB->get_record('prog_courseset', array('programid' => $this->program1->id, 'sortorder' => 3));

        // Try to get a courseset by id.
        $returnedcourseset1 = $progcontent->get_courseset_by_id($courseset1->id);
        // Check it contains the same values as well as data.
        $this->assertEquals(1, $returnedcourseset1->sortorder);
        $this->assertEquals(array($this->course1), $returnedcourseset1->get_courses());

        // Try to get a courseset by another id.
        $returnedcourseset3 = $progcontent->get_courseset_by_id($courseset3->id);
        // Check it contains the same values as well as data.
        $this->assertEquals(3, $returnedcourseset3->sortorder);
        $this->assertEquals(array($this->course3->id => $this->course3), $returnedcourseset3->get_courses());

        // Remove a courseset. We'll use the method which takes the sortorder in this case.
        $progcontent->delete_set(2);
        $progcontent->save_content();
        $returnedcourseset3 = $progcontent->get_courseset_by_id($courseset3->id);
        // The sortorder for the competency courseset should have changed.
        $this->assertEquals(2, $returnedcourseset3->sortorder);
        $this->assertEquals(array($this->course3->id => $this->course3), $returnedcourseset3->get_courses());

        // Now let's swap the order of the coursesets by moving the second one up.
        $progcontent->move_set_up(2);
        $progcontent->save_content();

        // Try to get a courseset by id.
        $returnedcourseset1 = $progcontent->get_courseset_by_id($courseset1->id);
        // Check it contains the new correct sortorder but still the same data such as courses.
        $this->assertEquals(2, $returnedcourseset1->sortorder);
        $this->assertEquals(array($this->course1), $returnedcourseset1->get_courses());

        // Try to get a courseset by another id.
        $returnedcourseset3 = $progcontent->get_courseset_by_id($courseset3->id);
        // Check it contains the new correct sortorder but still the same data such as courses.
        $this->assertEquals(1, $returnedcourseset3->sortorder);
        $this->assertEquals(array($this->course3->id => $this->course3), $returnedcourseset3->get_courses());
    }

    /**
     * Tests the delete_courseset_by_id method.
     */
    public function test_delete_courseset_by_id() {
        global $DB;

        $progcontent = new program_content($this->program1->id);
        $progcontent->add_set(program_content::CONTENTTYPE_MULTICOURSE);
        $progcontent->add_set(program_content::CONTENTTYPE_MULTICOURSE);
        $progcontent->add_set(program_content::CONTENTTYPE_COMPETENCY);

        /** @var course_set[] $coursesets */
        $coursesets = $progcontent->get_course_sets();

        // We'll add courses while we know what the sortorder is.
        $coursedata = new stdClass();
        $coursedata->{$coursesets[0]->get_set_prefix() . 'courseid'} = $this->course1->id;
        $progcontent->add_course(1, $coursedata);

        $coursedata->{$coursesets[1]->get_set_prefix() . 'courseid'} = $this->course2->id;
        $progcontent->add_course(2, $coursedata);

        /** @var \totara_hierarchy\testing\generator $hierarchygenerator */
        $hierarchygenerator = $this->generator->get_plugin_generator('totara_hierarchy');
        $competencyframework = $hierarchygenerator->create_comp_frame(array());
        $competencydata = array('frameworkid' => $competencyframework->id);
        $competency = $hierarchygenerator->create_comp($competencydata);
        // Completions for course 3 will be assigned to this competency.
        $course3evidenceid = $hierarchygenerator->assign_linked_course_to_competency($competency, $this->course3);

        // Add a competency to the competency courseset.
        $compdata = new stdClass();
        $compdata->{$coursesets[2]->get_set_prefix() . 'competencyid'} = $competency->id;
        $progcontent->add_competency(3, $compdata);

        // Now try to delete coursesets by id before saving using an arbitrary id number. This should return false.
        $this->assertEquals(false, $progcontent->delete_courseset_by_id(1));

        $progcontent->save_content();

        // Get the ids. At this stage we know what the sortorder should be and can test based on the below records.
        // Multi course set which contains course1.
        $courseset1 = $DB->get_record('prog_courseset', array('programid' => $this->program1->id, 'sortorder' => 1));
        // Multi course set which contains course2.
        $courseset2 = $DB->get_record('prog_courseset', array('programid' => $this->program1->id, 'sortorder' => 2));
        // Competency course set which contains competency 1 which links to course3.
        $courseset3 = $DB->get_record('prog_courseset', array('programid' => $this->program1->id, 'sortorder' => 3));

        $progcontent->delete_courseset_by_id($courseset1->id);

        // Check the deleted course set no longer gets returned.
        $this->assertFalse($progcontent->get_courseset_by_id($courseset1->id));

        // Check sortorders are updated.
        $returnedcourseset2 = $progcontent->get_courseset_by_id($courseset2->id);
        $this->assertEquals(1, $returnedcourseset2->sortorder);
        $returnedcourseset3 = $progcontent->get_courseset_by_id($courseset3->id);
        $this->assertEquals(2, $returnedcourseset3->sortorder);

        $progcontent->save_content();

        // Check correct prog_courseset record was deleted and none others.
        $this->assertFalse($DB->record_exists('prog_courseset', array('id' => $courseset1->id)));
        $this->assertTrue($DB->record_exists('prog_courseset', array('id' => $courseset2->id)));
        $this->assertTrue($DB->record_exists('prog_courseset', array('id' => $courseset3->id)));

        // Check that the correct prog_courseset_course record was deleted and none others.
        $this->assertFalse($DB->record_exists('prog_courseset_course', array('coursesetid' => $courseset1->id)));
        $this->assertTrue($DB->record_exists('prog_courseset_course', array('coursesetid' => $courseset2->id)));

        // Now delete again, this time with the competency course set.
        $progcontent->delete_courseset_by_id($courseset3->id);

        $this->assertFalse($progcontent->get_courseset_by_id($courseset3->id));

        // Check sortorders are correct.
        $returnedcourseset2 = $progcontent->get_courseset_by_id($courseset2->id);
        $this->assertEquals(1, $returnedcourseset2->sortorder);

        $progcontent->save_content();

        // Check correct prog_courseset record was deleted and none others.
        $this->assertTrue($DB->record_exists('prog_courseset', array('id' => $courseset2->id)));
        $this->assertFalse($DB->record_exists('prog_courseset', array('id' => $courseset3->id)));

        // No prog_courseset_course records should have been removed.
        $this->assertTrue($DB->record_exists('prog_courseset_course', array('coursesetid' => $courseset2->id)));
    }

    /**
     * Tests the contains_course method.
     */
    public function test_contains_course() {
        global $DB;

        // Set up program 1 first.
        $progcontent = new program_content($this->program1->id);
        $progcontent->add_set(program_content::CONTENTTYPE_MULTICOURSE);
        $progcontent->add_set(program_content::CONTENTTYPE_MULTICOURSE);
        $progcontent->add_set(program_content::CONTENTTYPE_COMPETENCY);

        /* @var course_set[] $coursesets */
        $coursesets = $progcontent->get_course_sets();

        // Add courses to course sets.
        $coursedata = new stdClass();
        $coursedata->{$coursesets[0]->get_set_prefix() . 'courseid'} = $this->course1->id;
        $progcontent->add_course(1, $coursedata);

        $coursedata->{$coursesets[1]->get_set_prefix() . 'courseid'} = $this->course2->id;
        $progcontent->add_course(2, $coursedata);

        /* @var \totara_hierarchy\testing\generator $hierarchygenerator */
        $hierarchygenerator = $this->generator->get_plugin_generator('totara_hierarchy');
        $competencyframework = $hierarchygenerator->create_comp_frame(array());
        $competencydata = array('frameworkid' => $competencyframework->id);
        $competency = $hierarchygenerator->create_comp($competencydata);
        // Completions for course 3 will be assigned to this competency.
        $course3evidenceid = $hierarchygenerator->assign_linked_course_to_competency($competency, $this->course3);

        // Add a competency to the competency courseset.
        $compdata = new stdClass();
        $compdata->{$coursesets[2]->get_set_prefix() . 'competencyid'} = $competency->id;
        $progcontent->add_competency(3, $compdata);

        $progcontent->save_content();

        // Then set up program 2.
        $progcontent = new program_content($this->program2->id);
        $progcontent->add_set(program_content::CONTENTTYPE_MULTICOURSE);

        /* @var course_set[] $coursesets */
        $coursesets = $progcontent->get_course_sets();

        // Add courses to course sets.
        $coursedata = new stdClass();
        $coursedata->{$coursesets[0]->get_set_prefix() . 'courseid'} = $this->course2->id;
        $progcontent->add_course(1, $coursedata);

        $progcontent->save_content();

        // Test the function.
        $program1 = new program($this->program1->id);
        $this->assertTrue($program1->content->contains_course($this->course1->id));
        $this->assertTrue($program1->content->contains_course($this->course2->id));
        $this->assertTrue($program1->content->contains_course($this->course3->id));
        $program2 = new program($this->program2->id);
        $this->assertFalse($program2->content->contains_course($this->course1->id));
        $this->assertTrue($program2->content->contains_course($this->course2->id));
        $this->assertFalse($program2->content->contains_course($this->course3->id));
    }

    public function test_delete_courseset_completion() {
        global $DB;

        set_config('enablecompletion', '1');

        $completion_generator = $this->generator->get_plugin_generator('core_completion');
        $completion_generator->enable_completion_tracking($this->course1);
        $completion_generator->enable_completion_tracking($this->course2);

        $this->program_generator->add_courses_and_courseset_to_program($this->program1, [[$this->course1], [$this->course2]]);

        $user1 = $this->generator->create_user();
        $user2 = $this->generator->create_user();
        $user3 = $this->generator->create_user();

        $this->program_generator->assign_program($this->program1->id, [$user1->id, $user2->id, $user3->id]);

        $completion_generator->complete_course($this->course1, $user1);
        $completion_generator->complete_course($this->course1, $user3);

        $coursesets = $this->program1->get_content()->get_course_sets();

        $this->assertEquals(8, $DB->count_records('prog_completion'));
        // Check we have two records for courseset 1 and 1 for courseset 2
        $this->assertEquals(3, $DB->count_records('prog_completion', ['coursesetid' => $coursesets[0]->id]));
        $this->assertEquals(2, $DB->count_records('prog_completion', ['coursesetid' => $coursesets[1]->id]));

        $this->program1->content->delete_set($coursesets[0]->sortorder);
        $this->program1->content->save_content();

        // Ensure we only deleted completion for courseset 1
        $this->assertEquals(0, $DB->count_records('prog_completion', ['coursesetid' => $coursesets[0]->id]));
        $this->assertEquals(2, $DB->count_records('prog_completion', ['coursesetid' => $coursesets[1]->id]));

        // Delete all program content
        $this->program1->content->delete();

        // Check that there are only 2 completion records left and they are both prog completion
        $this->assertEquals(3, $DB->count_records('prog_completion'));
        $this->assertEquals(3, $DB->count_records('prog_completion', ['coursesetid' => 0]));
    }

    public function test_clone() {
        global $DB;

        $this->assertEquals(0, $DB->count_records('prog_courseset'));
        $this->assertEquals(0, $DB->count_records('prog_courseset_course'));

        // Add content to program 1.
        $this->program_generator->add_courses_and_courseset_to_program($this->program1, [[$this->course1], [$this->course2], [$this->course3]]);
        $this->assertEquals(3, $DB->count_records('prog_courseset', ['programid' => $this->program1->id]));
        $this->assertEquals(3, $DB->count_records('prog_courseset'));
        $this->assertEquals(3, $DB->count_records('prog_courseset_course'));

        // Add content to program 2.
        $this->program_generator->add_courses_and_courseset_to_program($this->program2, [[$this->course1], [$this->course2], [$this->course3]]);
        $this->assertEquals(3, $DB->count_records('prog_courseset', ['programid' => $this->program1->id]));
        $this->assertEquals(6, $DB->count_records('prog_courseset'));
        $this->assertEquals(6, $DB->count_records('prog_courseset_course'));

        // Create a new program.
        $to_program = $this->program_generator->create_program();
        $this->assertEquals(0, $DB->count_records('prog_courseset', ['programid' => $to_program->id]));

        // Clone content from program 2 to the new program.
        $this->program2->get_content()->clone($to_program);
        $this->assertEquals(3, $DB->count_records('prog_courseset', ['programid' => $to_program->id]));
        $this->assertEquals(9, $DB->count_records('prog_courseset'));
        $this->assertEquals(9, $DB->count_records('prog_courseset_course'));

        // Ensure all the correct courses are cloned.
        $check_program = new program($this->program2->id);
        $course_sets = $check_program->get_content()->get_course_sets();
        $this->assertCount(3, $course_sets);
        $this->assertCount(1, $check_program->get_content()->get_course_sets()[0]->courses);
        $this->assertCount(1, $check_program->get_content()->get_course_sets()[1]->courses);
        $this->assertCount(1, $check_program->get_content()->get_course_sets()[2]->courses);
        $this->assertTrue($check_program->get_content()->get_course_sets()[0]->contains_course($this->course1->id));
        $this->assertTrue($check_program->get_content()->get_course_sets()[1]->contains_course($this->course2->id));
        $this->assertTrue($check_program->get_content()->get_course_sets()[02]->contains_course($this->course3->id));

        // Ensure programs are only cloned into a program with empty content.
        try {
            $this->program2->get_content()->clone($to_program);
        } catch (\coding_exception $ex) {
            $this->assertSame(
                'Coding error detected, it must be fixed by a programmer: Program content can only be cloned into a program with empty content',
                $ex->getMessage()
            );
        }
        $this->assertEquals(3, $DB->count_records('prog_courseset', ['programid' => $to_program->id]));
        $this->assertEquals(9, $DB->count_records('prog_courseset'));
        $this->assertEquals(9, $DB->count_records('prog_courseset_course'));
    }

}

