<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Ben Fesili <ben.fesili@totara.com>
 * @package totara_webapi
 */

use GraphQL\Error\DebugFlag;
use GraphQL\Executor\ExecutionResult;
use totara_api\global_api_config;
use totara_api\response_debug;
use totara_webapi\local\util;
use core_phpunit\testcase;

defined('MOODLE_INTERNAL') || die();

class totara_webapi_graphql_error_handler_test extends testcase {
    /**
     * @var array|null Registered client aware exceptions
     */
    private ?array $original = null;

    /**
     * Cleanup the changed client aware exceptions after the tests
     */
    protected function tearDown(): void {
        $reflect = new ReflectionClass(\totara_webapi\client_aware_exception_helper::class);
        $reflect->setStaticPropertyValue('registered_exceptions', $this->original);
        $this->original = null;

        parent::tearDown();
    }

    /**
     * Register client aware exceptions before the tests
     */
    protected function setUp(): void {
        parent::setUp();

        // Make sure our client-aware exception is registered
        $reflect = new ReflectionClass(\totara_webapi\client_aware_exception_helper::class);
        $value = $reflect->getStaticPropertyValue('registered_exceptions');
        $this->original = $value;
        $value[\totara_webapi\client_aware_exception::class] = [
            'category' => 'validation',
        ];
        $reflect->setStaticPropertyValue('registered_exceptions', $value);
    }

    /**
     * Check that normal mode removes the full path and replaces with dirroot
     * from setuplib's get_exception_info
     *
     * @return void
     */
    public function test_external_new_client_aware_exception_with_normal_mode(): void {
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_NORMAL, 'totara_api');

        $ex_message = __DIR__;
        $error = new Exception($ex_message, 0, new exception($ex_message));
        $expected_debug_message = $this->replace_dirroot($error);
        $scrubbed_errors = util::external_graphql_error_handler([$error], function($errors){
            static $debug = 1;
            return GraphQL\Error\FormattedError::prepareFormatter(null, $debug);
        });
        $exception_output = (reset($scrubbed_errors))($error);
        $this->assertSame("Internal server error", $exception_output['message']);
        $this->assertNotEquals($ex_message, $exception_output['message']);
        $this->assertSame($expected_debug_message, $exception_output['extensions']['debugMessage']);
    }

    /**
     * Check that none mode removes the full path and replaces with dirroot
     * from setuplib's get_exception_info
     *
     * @return void
     */
    public function test_external_new_client_aware_exception_with_none_mode(): void {
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_NONE, 'totara_api');

        $ex_message = __DIR__;
        $error = new Exception($ex_message, 0, new exception($ex_message));
        $scrubbed_errors = util::external_graphql_error_handler([$error], function($errors){
            static $debug = 0;
            return GraphQL\Error\FormattedError::prepareFormatter(null, $debug);
        });
        $exception_output = (reset($scrubbed_errors))($error);
        $this->assertSame("Internal server error", $exception_output['message']);
        $this->assertArrayNotHasKey('debugMessage', $exception_output);
    }

    /**
     * Check that debug mode doesn't replace and leaves the full path in
     *
     * @return void
     */
    public function test_external_test_new_client_aware_exception_with_debug_mode(): void {
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_DEVELOPER, 'totara_api');

        $ex_message = __DIR__;
        $error = new Exception($ex_message, 0, new exception($ex_message));
        $scrubbed_errors = util::external_graphql_error_handler([$error], function($errors){
            static $debug = 3;
            return GraphQL\Error\FormattedError::prepareFormatter(null, $debug);
        });
        $exception_output = (reset($scrubbed_errors))($error);
        $this->assertSame("Internal server error", $exception_output['message']);
        $this->assertArrayHasKey('trace', $exception_output['extensions']);
        $this->assertSame($ex_message, $exception_output['extensions']['debugMessage']);
    }

    /**
     * Assert the standard error handler hides/shows content based on the debugdeveloper flags for client aware exceptions.
     *
     * @return void
     */
    public function test_internal_client_aware(): void {
        // Client-aware exception, 'Standard error' is a safe message to share.
        $exception = new \totara_webapi\client_aware_exception(new Exception('Standard error'), ['category' => 'validation']);
        $result = new ExecutionResult(null, [$exception]);

        $result->setErrorsHandler(util::graphql_error_handler(...));
        $result->setErrorFormatter(util::graphql_error_formatter(...));

        // Assert - Debug Developer Disabled
        // Expect: We see the client-safe message, but no trace or lines.
        set_config('debugdeveloper', 0);
        $debug = DebugFlag::NONE;

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('Standard error', $error['message']);

        $this->assertArrayHasKey('extensions', $error);
        $this->assertEqualsCanonicalizing(['category' => 'validation'], $error['extensions']);
        $this->assertArrayNotHasKey('trace', $error['extensions']);
        $this->assertArrayNotHasKey('file', $error['extensions']);


        // Assert - Debug Developer Enabled
        // Expect: We see the client-safe message AND the trace and lines
        set_config('debugdeveloper', 1);
        $debug = DebugFlag::INCLUDE_DEBUG_MESSAGE | DebugFlag::INCLUDE_TRACE;

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('extensions', $error);
        $extensions = $error['extensions'];
        $this->assertArrayHasKey('category', $extensions);
        $this->assertEquals('validation', $extensions['category']);
        $this->assertArrayHasKey('trace', $extensions);
        $this->assertArrayHasKey('file', $extensions);
    }

    /**
     * Assert the standard error handler hides/shows content based on the debugdeveloper flags.
     *
     * @return void
     */
    public function test_internal_standard_exceptions(): void {
        // Normal exception - 'Standard error' is UNSAFE and should only show during debug
        $exception = new Exception('Standard error');
        $result = new ExecutionResult(null, [$exception]);
        $result->setErrorsHandler(util::graphql_error_handler(...));
        $result->setErrorFormatter(util::graphql_error_formatter(...));


        // Assert - Debug Developer Disabled
        // Expect: We see a standard message, no trace or lines
        set_config('debugdeveloper', 0);
        $debug = DebugFlag::NONE;

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('An error occurred', $error['message']);

        $this->assertArrayHasKey('extensions', $error);
        $this->assertEqualsCanonicalizing(['category' => 'internal'], $error['extensions']);
        $this->assertArrayNotHasKey('trace', $error['extensions']);
        $this->assertArrayNotHasKey('file', $error['extensions']);


        // Assert - Debug Developer Enabled
        // Expect: We see the standard message AND the trace and lines
        set_config('debugdeveloper', 1);
        $debug = DebugFlag::INCLUDE_DEBUG_MESSAGE | DebugFlag::INCLUDE_TRACE;

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('Internal server error', $error['message']);

        $this->assertArrayHasKey('extensions', $error);
        $extensions = $error['extensions'];
        $this->assertArrayHasKey('category', $extensions);
        $this->assertEquals('internal', $extensions['category']);
        $this->assertArrayHasKey('trace', $extensions);
        $this->assertArrayHasKey('file', $extensions);
        $this->assertArrayHasKey('debugMessage', $extensions);
        $this->assertEquals('Standard error', $extensions['debugMessage']);
    }

    /**
     * Assert the external API error handler hides/shows content based on the API flags.
     *
     * @return void
     */
    public function test_external_client_aware(): void {
        // Client-aware exception - 'Standard error with path' is SAFE
        $exception = new \totara_webapi\client_aware_exception(new Exception('Standard error with path ' . __FILE__), ['category' => 'validation']);
        $result = new ExecutionResult(null, [$exception]);
        $result->setErrorsHandler(util::external_graphql_error_handler(...));
        $result->setErrorFormatter(util::external_graphql_error_formatter(...));

        // Assert - response_debug = NONE
        // Expect: We see the safe message, but with no trace or file paths
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_NONE, 'totara_api');
        $debug = global_api_config::get_response_debug_flag(response_debug::ERROR_RESPONSE_LEVEL_NONE);

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('Standard error with path [dirroot]/totara/webapi/tests/graphql_error_handler_test.php', $error['message']);

        $this->assertArrayHasKey('extensions', $error);
        $this->assertEqualsCanonicalizing(['category' => 'validation'], $error['extensions']);
        $this->assertArrayNotHasKey('trace', $error['extensions']);
        $this->assertArrayNotHasKey('file', $error['extensions']);


        // Assert - response_debug = NORMAL
        // Expect: We see the safe message, no trace or lines
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_NORMAL, 'totara_api');
        $debug = global_api_config::get_response_debug_flag(response_debug::ERROR_RESPONSE_LEVEL_NORMAL);

        $exception = new \totara_webapi\client_aware_exception(new Exception('Standard error with path ' . __FILE__), ['category' => 'validation']);
        $result = new ExecutionResult(null, [$exception]);
        $result->setErrorsHandler(util::external_graphql_error_handler(...));
        $result->setErrorFormatter(util::external_graphql_error_formatter(...));

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('Standard error with path [dirroot]/totara/webapi/tests/graphql_error_handler_test.php', $error['message']);

        $this->assertArrayHasKey('extensions', $error);
        $this->assertEqualsCanonicalizing(['category' => 'validation'], $error['extensions']);
        $this->assertArrayNotHasKey('trace', $error['extensions']);
        $this->assertArrayNotHasKey('file', $error['extensions']);


        // Assert - response_debug = Developer
        // Expect: We see everything
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_DEVELOPER, 'totara_api');
        $debug = global_api_config::get_response_debug_flag(response_debug::ERROR_RESPONSE_LEVEL_DEVELOPER);

        $exception = new \totara_webapi\client_aware_exception(new Exception('Standard error with path ' . __FILE__), ['category' => 'validation']);
        $result = new ExecutionResult(null, [$exception]);
        $result->setErrorsHandler(util::external_graphql_error_handler(...));
        $result->setErrorFormatter(util::external_graphql_error_formatter(...));

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertStringContainsString('Standard error with path ', $error['message']);
        $this->assertStringContainsString('/totara/webapi/tests/graphql_error_handler_test.php', $error['message']);
        $this->assertStringNotContainsString('[dirroot]', $error['message']);

        // Content is in the main message, we don't have a debug for client-aware
        $this->assertArrayHasKey('extensions', $error);
        $extensions = $error['extensions'];
        $this->assertArrayNotHasKey('debugMessage', $extensions);
        $this->assertArrayHasKey('trace', $error['extensions']);
        $this->assertArrayHasKey('file', $error['extensions']);
    }

    /**
     * Assert the external API error handler hides/shows content based on the API flags.
     *
     * @return void
     */
    public function test_external_standard_exceptions(): void {
        // Normal exception - 'Standard error with path' is UNSAFE and should only show if debug is on
        $exception = new Exception('Standard error with path ' . __FILE__);
        $result = new ExecutionResult(null, [$exception]);
        $result->setErrorsHandler(util::external_graphql_error_handler(...));
        $result->setErrorFormatter(util::external_graphql_error_formatter(...));


        // Assert - response_debug = NONE
        // Expect: We see a suppressed message, no trace or lines
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_NONE, 'totara_api');
        $debug = global_api_config::get_response_debug_flag(response_debug::ERROR_RESPONSE_LEVEL_NONE);

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('Internal server error', $error['message']);

        $this->assertArrayHasKey('extensions', $error);
        $this->assertEqualsCanonicalizing(['category' => 'internal'], $error['extensions']);
        $this->assertArrayNotHasKey('trace', $error['extensions']);
        $this->assertArrayNotHasKey('file', $error['extensions']);


        // Assert - response_debug = NORMAL
        // Expect: We see the sanitsized message, no trace or lines
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_NORMAL, 'totara_api');
        $debug = global_api_config::get_response_debug_flag(response_debug::ERROR_RESPONSE_LEVEL_NORMAL);

        $exception = new Exception('Standard error with path ' . __FILE__);
        $result = new ExecutionResult(null, [$exception]);
        $result->setErrorsHandler(util::external_graphql_error_handler(...));
        $result->setErrorFormatter(util::external_graphql_error_formatter(...));

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('Internal server error', $error['message']);

        // Content in the debugMessage
        $this->assertArrayHasKey('extensions', $error);
        $extensions = $error['extensions'];
        $this->assertArrayHasKey('debugMessage', $extensions);
        $this->assertEquals('Standard error with path [dirroot]/totara/webapi/tests/graphql_error_handler_test.php', $extensions['debugMessage']);

        $this->assertArrayNotHasKey('trace', $error['extensions']);
        $this->assertArrayNotHasKey('file', $error['extensions']);


        // Assert - response_debug = Developer
        // Expect: We see everything
        set_config('response_debug', response_debug::ERROR_RESPONSE_LEVEL_DEVELOPER, 'totara_api');
        $debug = global_api_config::get_response_debug_flag(response_debug::ERROR_RESPONSE_LEVEL_DEVELOPER);

        $exception = new Exception('Standard error with path ' . __FILE__);
        $result = new ExecutionResult(null, [$exception]);
        $result->setErrorsHandler(util::external_graphql_error_handler(...));
        $result->setErrorFormatter(util::external_graphql_error_formatter(...));

        $output = $result->toArray($debug);
        $this->assertArrayHasKey('errors', $output);
        $this->assertCount(1, $output['errors']);
        $error = $output['errors'][0];

        $this->assertArrayHasKey('message', $error);
        $this->assertEquals('Internal server error', $error['message']);

        // Content in the debugMessage
        $this->assertArrayHasKey('extensions', $error);
        $extensions = $error['extensions'];
        $this->assertArrayHasKey('debugMessage', $extensions);
        $this->assertStringContainsString('Standard error with path', $extensions['debugMessage']);
        $this->assertStringContainsString('/totara/webapi/tests/graphql_error_handler_test.php', $extensions['debugMessage']);
        $this->assertStringNotContainsString('[dirroot]', $extensions['debugMessage']);

        $this->assertArrayHasKey('trace', $error['extensions']);
        $this->assertArrayHasKey('file', $error['extensions']);
    }

    private function replace_dirroot(Exception $exception): string {
        global $CFG;
        $search = "";
        $replace = "";
        if (property_exists($CFG, 'dirroot')) {
            $search = $CFG->dirroot;
            $replace = "[dirroot]";
        }
        return str_replace($search, $replace, $exception->getMessage());
    }
}
