<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * myprofile renderer.
 *
 * @package    core_user
 * @copyright  2015 onwards Ankit Agarwal <ankit.agrr@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace core_user\output\myprofile;
defined('MOODLE_INTERNAL') || die;
/**
 * Report log renderer's for printing reports.
 *
 * @since      Moodle 2.9
 * @package    core_user
 * @copyright  2015 onwards Ankit Agarwal <ankit.agrr@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class renderer extends \plugin_renderer_base {
    /**
     * Render the whole tree.
     *
     * @param tree $tree
     *
     * @return string
     */
    public function render_tree(tree $tree) {
        $return = \html_writer::start_tag('div', array('class' => 'profile_tree'));
        $categories = $tree->categories;
        foreach ($categories as $category) {
            $return .= $this->render($category);
        }
        $return .= \html_writer::end_tag('div');
        return $return;
    }

    /**
     * Render a category.
     *
     * @param category $category
     *
     * @return string
     */
    public function render_category(category $category) {
        $classes = $category->classes;
        if (empty($classes)) {
            $return = \html_writer::start_tag('section', array('class' => 'node_category'));
        } else {
            $return = \html_writer::start_tag('section', array('class' => 'node_category ' . $classes));
        }
        $return .= \html_writer::tag('h3', $category->title);
        $nodes = $category->nodes;
        if (empty($nodes)) {
            // No nodes, nothing to render.
            return '';
        }

        $return .= \html_writer::start_tag('div', array('class' => 'userdetails'));
        $output_nodes = [];
        $order = 0;
        // Render nodes with order
        foreach ($nodes as $node) {
            if ($node->content === null || $node->content === '') {
                $order += 1;
                if (empty($output_nodes[$order]['non-node'])) {
                    $output_nodes[$order]['non-node'] = $this->render($node);
                } else {
                    $output_nodes[$order]['non-node'] .= $this->render($node);
                }
            } else {
                if (empty($output_nodes[$order]['node'])) {
                    $output_nodes[$order]['node'] = $this->render($node);
                } else {
                    $output_nodes[$order]['node'] .= $this->render($node);
                }
            }
        }
        // Wrap nodes with order
        foreach ($output_nodes as $node) {
            if (!empty($node['non-node'])) {
                $return .= $node['non-node'];
            }
            if (!empty($node['node'])) {
                $return .= \html_writer::tag('dl', $node['node'], array('class' => 'contentnode__wrapper'));
            }
        }

        $return .= \html_writer::end_tag('div');
        $return .= \html_writer::end_tag('section');
        return $return;
    }

    /**
     * Render a node.
     *
     * @param node $node
     *
     * @return string
     */
    public function render_node(node $node) {
        $return = '';
        if (is_object($node->url)) {
            $header = \html_writer::link($node->url, $node->title);
        } else {
            $header = $node->title;
        }
        $icon = $node->icon;
        if (!empty($icon)) {
            $header .= $this->render($icon);
        }
        $content = $node->content;
        $classes = $node->classes;
        if ($content !== null && $content !== "") {
            // There is some content to display below this make this a header.
            $return = \html_writer::tag('dt', $header, array('class' => 'contentnode__header'));
            $return .= \html_writer::tag('dd', $content);

            if ($classes) {
                $return = \html_writer::tag('div', $return, array('class' => 'contentnode ' . $classes));
            } else {
                $return = \html_writer::tag('div', $return, array('class' => 'contentnode'));
            }
        } else {
            $return = \html_writer::span($header);
            $return = \html_writer::tag('div', $return, array('class' => $classes));
        }

        return $return;
    }
}
