<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Text profile field.
 *
 * @package    profilefield_text
 * @copyright  2007 onwards Shane Elliot {@link http://pukunui.com}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Class profile_field_text
 *
 * @copyright  2007 onwards Shane Elliot {@link http://pukunui.com}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class profile_field_text extends profile_field_base {

    /**
     * Overwrite the base class to display the data for this field
     */
    public function display_data() {
        // Default formatting.
        $data = parent::display_data();

        // Are we creating a link?
        if (!empty($this->field->param4) and !empty($data)) {

            // Define the target.
            if (! empty($this->field->param5)) {
                $target = 'target="'.$this->field->param5.'"';
            } else {
                $target = '';
            }

            // Create the link.
            $data = clean_text('<a href="'.str_replace('$$', urlencode($data), $this->field->param4).'" '.$target.'>'.htmlspecialchars($data).'</a>');
        }

        return $data;
    }

    /**
     * Add fields for editing a text profile field.
     * @param moodleform $mform
     */
    public function edit_field_add($mform) {
        $size = $this->field->param1;
        $maxlength = $this->field->param2;
        $fieldtype = ($this->field->param3 == 1 ? 'password' : 'text');

        // Create the form field.
        if ($this->is_locked() and !has_capability('moodle/user:update', context_system::instance())) {
            $data = parent::display_data();
            $mform->addElement(
                'static',
                'freezedisplay',
                format_string($this->field->name),
                html_writer::div(
                    !empty($data) ? format_text($data, FORMAT_MOODLE) : get_string('readonlyemptyfield', 'profilefield_text'),
                    null,
                    ['id' => 'id_field_' . $this->field->name]
                )
            );

            // This is a static field. If the field is required skip the required rule to avoid error.
            if ($this->is_required()) {
                $this->skiprequired = true;
            }

            return;
        }
        $mform->addElement($fieldtype, $this->inputname, format_string($this->field->name), 'maxlength="'.$maxlength.'" size="'.$size.'" ');
        $mform->setType($this->inputname, PARAM_TEXT);
    }

    /**
     * @inheritDoc
     */
    function validate_field_from_inputs(array $params): void {
        $field = $this->field->shortname;

        if (isset($params['data_format'])) {
            throw new moodle_exception("{$field}: data_format should not be passed for text fields.");
        }

        $maxlength = $this->field->param2;
        if (!empty(trim($params['data'])) && core_text::strlen($params['data']) > $maxlength) {
            throw new moodle_exception("{$field}: The data must have less than {$maxlength} characters.");
        }

        parent::validate_field_from_inputs($params);
    }

}