<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Rest server tests.
 *
 * @package    webservice_rest
 * @copyright  2016 Frédéric Massart - FMCorz.net
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use PHPUnit\Framework\MockObject\MockObject;

defined('MOODLE_INTERNAL') || die();
global $CFG;

require_once($CFG->libdir . '/externallib.php');
require_once($CFG->dirroot . '/webservice/rest/locallib.php');

/**
 * Rest server testcase.
 *
 * @package    webservice_rest
 * @copyright  2016 Frédéric Massart - FMCorz.net
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class webservice_rest_server_test extends \core_phpunit\testcase {

    /**
     * Data provider for test_xmlize.
     * @return array
     */
    public static function xmlize_provider() {
        $data = [];
        $data[] = [null, null, ''];
        $data[] = [new external_value(PARAM_BOOL), false, "<VALUE>0</VALUE>\n"];
        $data[] = [new external_value(PARAM_BOOL), true, "<VALUE>1</VALUE>\n"];
        $data[] = [new external_value(PARAM_ALPHA), null, "<VALUE null=\"null\"/>\n"];
        $data[] = [new external_value(PARAM_ALPHA), 'a', "<VALUE>a</VALUE>\n"];
        $data[] = [new external_value(PARAM_INT), 123, "<VALUE>123</VALUE>\n"];
        $data[] = [
            new external_multiple_structure(new external_value(PARAM_INT)),
            [1, 2, 3],
            "<MULTIPLE>\n" .
            "<VALUE>1</VALUE>\n" .
            "<VALUE>2</VALUE>\n" .
            "<VALUE>3</VALUE>\n" .
            "</MULTIPLE>\n"
        ];
        $data[] = [ // Multiple structure with null value.
            new external_multiple_structure(new external_value(PARAM_ALPHA)),
            ['A', null, 'C'],
            "<MULTIPLE>\n" .
            "<VALUE>A</VALUE>\n" .
            "<VALUE null=\"null\"/>\n" .
            "<VALUE>C</VALUE>\n" .
            "</MULTIPLE>\n"
        ];
        $data[] = [ // Multiple structure without values.
            new external_multiple_structure(new external_value(PARAM_ALPHA)),
            [],
            "<MULTIPLE>\n" .
            "</MULTIPLE>\n"
        ];
        $data[] = [
            new external_single_structure([
                'one' => new external_value(PARAM_INT),
                'two' => new external_value(PARAM_INT),
                'three' => new external_value(PARAM_INT),
            ]),
            ['one' => 1, 'two' => 2, 'three' => 3],
            "<SINGLE>\n" .
            "<KEY name=\"one\"><VALUE>1</VALUE>\n</KEY>\n" .
            "<KEY name=\"two\"><VALUE>2</VALUE>\n</KEY>\n" .
            "<KEY name=\"three\"><VALUE>3</VALUE>\n</KEY>\n" .
            "</SINGLE>\n"
        ];
        $data[] = [ // Single structure with null value.
            new external_single_structure([
                'one' => new external_value(PARAM_INT),
                'two' => new external_value(PARAM_INT),
                'three' => new external_value(PARAM_INT),
            ]),
            ['one' => 1, 'two' => null, 'three' => 3],
            "<SINGLE>\n" .
            "<KEY name=\"one\"><VALUE>1</VALUE>\n</KEY>\n" .
            "<KEY name=\"two\"><VALUE null=\"null\"/>\n</KEY>\n" .
            "<KEY name=\"three\"><VALUE>3</VALUE>\n</KEY>\n" .
            "</SINGLE>\n"
        ];
        $data[] = [ // Single structure missing keys.
            new external_single_structure([
                'one' => new external_value(PARAM_INT),
                'two' => new external_value(PARAM_INT),
                'three' => new external_value(PARAM_INT),
            ]),
            ['two' => null, 'three' => 3],
            "<SINGLE>\n" .
            "<KEY name=\"one\"><VALUE null=\"null\"/>\n</KEY>\n" .
            "<KEY name=\"two\"><VALUE null=\"null\"/>\n</KEY>\n" .
            "<KEY name=\"three\"><VALUE>3</VALUE>\n</KEY>\n" .
            "</SINGLE>\n"
        ];
        $data[] = [ // Nested structure.
            new external_single_structure([
                'one' => new external_multiple_structure(
                    new external_value(PARAM_INT)
                ),
                'two' => new external_multiple_structure(
                    new external_single_structure([
                        'firstname' => new external_value(PARAM_RAW),
                        'lastname' => new external_value(PARAM_RAW),
                    ])
                ),
                'three' => new external_single_structure([
                    'firstname' => new external_value(PARAM_RAW),
                    'lastname' => new external_value(PARAM_RAW),
                ]),
            ]),
            [
                'one' => [2, 3, 4],
                'two' => [
                    ['firstname' => 'Louis', 'lastname' => 'Armstrong'],
                    ['firstname' => 'Neil', 'lastname' => 'Armstrong'],
                ],
                'three' => ['firstname' => 'Neil', 'lastname' => 'Armstrong'],
            ],
            "<SINGLE>\n" .
            "<KEY name=\"one\"><MULTIPLE>\n".
                "<VALUE>2</VALUE>\n" .
                "<VALUE>3</VALUE>\n" .
                "<VALUE>4</VALUE>\n" .
            "</MULTIPLE>\n</KEY>\n" .
            "<KEY name=\"two\"><MULTIPLE>\n".
                "<SINGLE>\n" .
                    "<KEY name=\"firstname\"><VALUE>Louis</VALUE>\n</KEY>\n" .
                    "<KEY name=\"lastname\"><VALUE>Armstrong</VALUE>\n</KEY>\n" .
                "</SINGLE>\n" .
                "<SINGLE>\n" .
                    "<KEY name=\"firstname\"><VALUE>Neil</VALUE>\n</KEY>\n" .
                    "<KEY name=\"lastname\"><VALUE>Armstrong</VALUE>\n</KEY>\n" .
                "</SINGLE>\n" .
            "</MULTIPLE>\n</KEY>\n" .
            "<KEY name=\"three\"><SINGLE>\n" .
                "<KEY name=\"firstname\"><VALUE>Neil</VALUE>\n</KEY>\n" .
                "<KEY name=\"lastname\"><VALUE>Armstrong</VALUE>\n</KEY>\n" .
            "</SINGLE>\n</KEY>\n" .
            "</SINGLE>\n"
        ];
        $data[] = [ // Nested structure with missing keys.
            new external_single_structure([
                'one' => new external_multiple_structure(
                    new external_value(PARAM_INT)
                ),
                'two' => new external_multiple_structure(
                    new external_single_structure([
                        'firstname' => new external_value(PARAM_RAW),
                        'lastname' => new external_value(PARAM_RAW),
                    ])
                ),
                'three' => new external_single_structure([
                    'firstname' => new external_value(PARAM_RAW),
                    'lastname' => new external_value(PARAM_RAW),
                ]),
            ]),
            [
                'two' => [
                    ['firstname' => 'Louis'],
                    ['lastname' => 'Armstrong'],
                ],
                'three' => ['lastname' => 'Armstrong'],
            ],
            "<SINGLE>\n" .
            "<KEY name=\"one\"><MULTIPLE>\n</MULTIPLE>\n</KEY>\n" .
            "<KEY name=\"two\"><MULTIPLE>\n".
                "<SINGLE>\n" .
                    "<KEY name=\"firstname\"><VALUE>Louis</VALUE>\n</KEY>\n" .
                    "<KEY name=\"lastname\"><VALUE null=\"null\"/>\n</KEY>\n" .
                "</SINGLE>\n" .
                "<SINGLE>\n" .
                    "<KEY name=\"firstname\"><VALUE null=\"null\"/>\n</KEY>\n" .
                    "<KEY name=\"lastname\"><VALUE>Armstrong</VALUE>\n</KEY>\n" .
                "</SINGLE>\n" .
            "</MULTIPLE>\n</KEY>\n" .
            "<KEY name=\"three\"><SINGLE>\n" .
                "<KEY name=\"firstname\"><VALUE null=\"null\"/>\n</KEY>\n" .
                "<KEY name=\"lastname\"><VALUE>Armstrong</VALUE>\n</KEY>\n" .
            "</SINGLE>\n</KEY>\n" .
            "</SINGLE>\n"
        ];
        return $data;
    }

    /**
     * @dataProvider xmlize_provider
     * @param external_description $description The data structure.
     * @param mixed $value The value to xmlise.
     * @param mixed $expected The expected output.
     */
    public function test_xmlize($description, $value, $expected) {
        $method = new ReflectionMethod('webservice_rest_server', 'xmlize_result');
        $method->setAccessible(true);
        $this->assertEquals($expected, $method->invoke(null, $value, $description));
    }

    public function test_parse_request_retrieves_username_and_password() {
        $server = $this->get_mocked_server(WEBSERVICE_AUTHMETHOD_USERNAME);

        $server_reflection = new ReflectionClass($server);
        $username = $server_reflection->getProperty('username');
        $username->setAccessible(true);
        $password = $server_reflection->getProperty('password');
        $password->setAccessible(true);

        $this->callInternalMethod($server, 'parse_request', []);

        $this->assertNull($username->getValue($server));
        $this->assertNull($password->getValue($server));

        $_GET['wsusername'] = 'this username should not show as using _GET is naughty!';
        $_GET['wspassword'] = 'this password should not show as using _GET is naughty!';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertNull($username->getValue($server));
        $this->assertNull($password->getValue($server));

        $_POST['wsusername'] = 'my username!';
        $_POST['wspassword'] = 'my secure password';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('my username!', $username->getValue($server));
        $this->assertEquals('my secure password', $password->getValue($server));
    }

    /**
     * Test the $CFG->revert_TL_42916_until_t20 flag works as intended.
     * @return void
     */
    public function test_parse_request_retrieves_username_and_password_revert_TL_42916_until_t20() {
        global $CFG;
        $CFG->revert_TL_42916_until_t20 = true;

        $server = $this->get_mocked_server(WEBSERVICE_AUTHMETHOD_USERNAME);

        $server_reflection = new ReflectionClass($server);
        $username = $server_reflection->getProperty('username');
        $username->setAccessible(true);
        $password = $server_reflection->getProperty('password');
        $password->setAccessible(true);

        $this->callInternalMethod($server, 'parse_request', []);

        $this->assertNull($username->getValue($server));
        $this->assertNull($password->getValue($server));

        $_POST['wsusername'] = 'my username from POST';
        $_POST['wspassword'] = 'my password from POST';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('my username from POST', $username->getValue($server));
        $this->assertEquals('my password from POST', $password->getValue($server));

        unset($_POST['wsusername']);
        unset($_POST['wspassword']);

        $_GET['wsusername'] = 'my username from GET';
        $_GET['wspassword'] = 'my password from GET';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('my username from GET', $username->getValue($server));
        $this->assertEquals('my password from GET', $password->getValue($server));

        unset($CFG->revert_TL_42916_until_t20);
    }


    public function test_parse_request_retrieves_token() {
        $server = $this->get_mocked_server(WEBSERVICE_AUTHMETHOD_PERMANENT_TOKEN);

        $server_reflection = new ReflectionClass($server);
        $token = $server_reflection->getProperty('token');
        $token->setAccessible(true);

        $this->callInternalMethod($server, 'parse_request', []);

        $this->assertNull($token->getValue($server));

        $_GET['wstoken'] = 'mytoken should not show as using _GET is naughty!';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertNull($token->getValue($server));

        $_POST['wstoken'] = 'mytoken';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('mytoken', $token->getValue($server));
    }

    /**
     * Test the $CFG->revert_TL_42916_until_t20 flag works as intended.
     * @return void
     */
    public function test_parse_request_retrieves_token_revert_TL_42916_until_t20(): void {
        global $CFG;
        $CFG->revert_TL_42916_until_t20 = true;

        $server = $this->get_mocked_server(WEBSERVICE_AUTHMETHOD_PERMANENT_TOKEN);

        $server_reflection = new ReflectionClass($server);
        $token = $server_reflection->getProperty('token');
        $token->setAccessible(true);

        $this->callInternalMethod($server, 'parse_request', []);

        $this->assertNull($token->getValue($server));

        $_POST['wstoken'] = 'mytoken from POST';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('mytoken from POST', $token->getValue($server));
        unset($_POST['wstoken']);

        $_GET['wstoken'] = 'mytoken from GET';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('mytoken from GET', $token->getValue($server));

        unset($CFG->revert_TL_42916_until_t20);
    }

    /**
     * @param int $auth_method authentication method of the web service (WEBSERVICE_AUTHMETHOD_PERMANENT_TOKEN, ...)
     * @return (webservice_rest_server&MockObject)
     */
    public function get_mocked_server(int $auth_method): MockObject {
        return $this->getMockBuilder('\webservice_rest_server')
            ->setConstructorArgs([$auth_method])
            ->getMock();
    }

}
