<?php
/**
 * This file is part of Totara Core
 *
 * Copyright (C) 2025 onwards Totara Learning Solutions LTD
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Aaron Machin <aaron.machin@totara.com>
 * @package webservice_soap
 */

defined('MOODLE_INTERNAL') || die();

use core_phpunit\testcase;
use PHPUnit\Framework\MockObject\MockObject;

global $CFG;

require_once($CFG->libdir . '/externallib.php');
require_once($CFG->dirroot . '/webservice/soap/locallib.php');

class webservice_soap_server_test extends testcase {

    public function test_parse_request_retrieves_username_and_password_() {
        global $CFG;

        // Workaround as this is required by \get_file_argument which is called within parse_request().
        $_SERVER = [
            'SERVER_SOFTWARE' => '',
        ];

        $server = $this->get_mocked_server(WEBSERVICE_AUTHMETHOD_USERNAME);

        $server_reflection = new ReflectionClass($server);
        $username = $server_reflection->getProperty('username');
        $username->setAccessible(true);
        $password = $server_reflection->getProperty('password');
        $password->setAccessible(true);

        $this->callInternalMethod($server, 'parse_request', []);

        $this->assertNull($username->getValue($server));
        $this->assertNull($password->getValue($server));

        $_POST['wsusername'] = 'my username from POST';
        $_POST['wspassword'] = 'my password from POST';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('my username from POST', $username->getValue($server));
        $this->assertEquals('my password from POST', $password->getValue($server));

        unset($_POST['wsusername']);
        unset($_POST['wspassword']);

        $_GET['wsusername'] = 'my username from GET';
        $_GET['wspassword'] = 'my password from GET';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('my username from GET', $username->getValue($server));
        $this->assertEquals('my password from GET', $password->getValue($server));
    }

    public function test_parse_request_retrieves_token(): void {
        global $CFG;

        $server = $this->get_mocked_server(WEBSERVICE_AUTHMETHOD_PERMANENT_TOKEN);

        $server_reflection = new ReflectionClass($server);
        $token = $server_reflection->getProperty('token');
        $token->setAccessible(true);

        $this->callInternalMethod($server, 'parse_request', []);

        $this->assertNull($token->getValue($server));

        $_POST['wstoken'] = 'mytoken from POST';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('mytoken from POST', $token->getValue($server));
        unset($_POST['wstoken']);

        $_GET['wstoken'] = 'mytoken from GET';

        $this->callInternalMethod($server, 'parse_request', []);
        $this->assertEquals('mytoken from GET', $token->getValue($server));
    }

    /**
     * @param int $auth_method authentication method of the web service (WEBSERVICE_AUTHMETHOD_PERMANENT_TOKEN, ...)
     * @return (webservice_soap_server&MockObject)
     */
    public function get_mocked_server(int $auth_method): MockObject {
        return $this->getMockBuilder('\webservice_soap_server')
            ->setConstructorArgs([$auth_method])
            ->getMock();
    }
}
